package cn.tdchain.api;

import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSONObject;

import cn.tdchain.api.rpc.CeConnection;
import cn.tdchain.api.rpc.CeConnectionUtils;
import cn.tdchain.api.service.ContractService;
import cn.tdchain.api.service.impl.ContractServiceImpl;
import cn.tdchain.cb.constant.ResultConstants;
import cn.tdchain.cb.domain.ContractState;
import cn.tdchain.cb.exception.BusinessException;
import cn.tdchain.cb.util.JsonUtils;
import cn.tdchain.cb.util.StringUtils;
import cn.tdchain.jbcc.rpc.RPCMessage;

/**
 * Contract API.
 * 
 * @version 1.0
 * @author Bingoer.H 2018-12-18
 */
public class ContractApi {

    private static final String CONTRACT_NAME = "contractName";
    private static final String KS_PASSWORD = "ksPassword";
    private static final String ADDRESS = "address";
//    private static final Logger log = LogManager.getLogger("TD_API");
    private ContractService contractService = new ContractServiceImpl();
    private CeConnection con = CeConnectionUtils.getConnection();

    /**
     * Create contract.
     * 
     * @param address operator address
     * @param ksPassword operator key store password
     * @param contractName contract name
     * @param templateName template hash
     * @param args constructor arguments
     * @return contract hash
     * @throws BusinessException business exception
     */
    public Map<String, String> create(String address, String ksPassword,
                                      String contractName, String templateName,
                                      String[] args)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(contractName)
                || StringUtils.isBlank(templateName) || args == null
                || args.length == 0) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put(ADDRESS, address);
        msg.getCommand().put(KS_PASSWORD, ksPassword);
        msg.getCommand().put(CONTRACT_NAME, contractName);
        msg.getCommand().put("templateName", templateName);
        msg.getCommand().put("args", JsonUtils.toJson(args));
        msg.setTargetType(RPCMessage.TargetType.CREATE_CONTRACT);
        return con.sendAndReturn(msg, ResultConstants.CONTRACT_CREATE_FAILED);
    }

    /**
     * Run contract.
     * 
     * @param address account address
     * @param ksPassword account key store password
     * @param contractName contract name
     * @param methodName contract method name
     * @param args method arguments
     * @return record hash
     * @throws BusinessException business exception
     */
    public Map<String, String> run(String address, String ksPassword,
                                   String contractName, String methodName,
                                   String[] args)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(contractName)
                || StringUtils.isBlank(methodName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        if (methodName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, methodName));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put(ADDRESS, address);
        msg.getCommand().put(KS_PASSWORD, ksPassword);
        msg.getCommand().put(CONTRACT_NAME, contractName);
        msg.getCommand().put("methodName", methodName);
        msg.getCommand().put("args", JsonUtils.toJson(args));
        msg.setTargetType(RPCMessage.TargetType.RUN_CONTRACT);
        return con.sendAndReturn(msg, ResultConstants.CONTRACT_RUN_FAILED);
    }

    /**
     * Freeze contract.
     * 
     * @param address contract owner or super account address
     * @param ksPassword account key store password
     * @param contractName contract name
     * @return record hash
     * @throws BusinessException business exception
     */
    public Map<String, String> freeze(String address, String ksPassword,
                                      String contractName)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(contractName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put(ADDRESS, address);
        msg.getCommand().put(KS_PASSWORD, ksPassword);
        msg.getCommand().put(CONTRACT_NAME, contractName);
        msg.setTargetType(RPCMessage.TargetType.FREEZE_CONTRACT);
        return con.sendAndReturn(msg, ResultConstants.CONTRACT_FREEZE_FAILED);
    }

    /**
     * Unfreeze contract.
     * 
     * @param address owner or super account address
     * @param ksPassword account key store password
     * @param contractName contract name
     * @return result message
     * @throws BusinessException business exception
     */
    public Map<String, String> unfreeze(String address, String ksPassword,
                                        String contractName)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(contractName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put(ADDRESS, address);
        msg.getCommand().put(KS_PASSWORD, ksPassword);
        msg.getCommand().put(CONTRACT_NAME, contractName);
        msg.setTargetType(RPCMessage.TargetType.UNFREEZE_CONTRACT);
        return con.sendAndReturn(msg, ResultConstants.CONTRACT_UNFREEZE_FAILED);
    }

    /**
     * Update contract.
     * 
     * @param address contract owner address
     * @param ksPassword owner account key store password
     * @param contractName contract name
     * @param syncAttrs synchronize attributes
     * @param newArgs new constructor arguments
     * @return new contract hash or error message
     * @throws BusinessException business exception
     */
    public Map<String, String> update(String address, String ksPassword,
                                      String contractName, String[] syncAttrs,
                                      String[] newArgs)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(contractName) || newArgs == null
                || newArgs.length == 0) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put(ADDRESS, address);
        msg.getCommand().put(KS_PASSWORD, ksPassword);
        msg.getCommand().put(CONTRACT_NAME, contractName);
        msg.getCommand().put("syncAttrs", JsonUtils.toJson(syncAttrs));
        msg.getCommand().put("newArgs", JsonUtils.toJson(newArgs));
        msg.setTargetType(RPCMessage.TargetType.UPDATE_CONTRACT);
        return con.sendAndReturn(msg, ResultConstants.CONTRACT_UPDATE_FAILED);
    }

    /**
     * Migration contract to new template.
     * 
     * @param address operator account address
     * @param ksPassword super account key store password
     * @param contractName contract name
     * @param newTemplateName new template name
     * @return contract hash
     * @throws BusinessException business exception
     */
    public Map<String, String> migrateToNewTemplate(String address,
                                                    String ksPassword,
                                                    String contractName,
                                                    String newTemplateName)
        throws BusinessException {
        if (StringUtils.isBlank(address) || StringUtils.isBlank(ksPassword)
                || StringUtils.isBlank(contractName)
                || StringUtils.isBlank(newTemplateName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put(ADDRESS, address);
        msg.getCommand().put(KS_PASSWORD, ksPassword);
        msg.getCommand().put(CONTRACT_NAME, contractName);
        msg.getCommand().put("newTemplateName", newTemplateName);
        msg.setTargetType(RPCMessage.TargetType.MIGRATE_CONTRACT);
        return con.sendAndReturn(msg,
                ResultConstants.CONTRACT_MIGRATION_FAILED);
    }

    /**
     * Find contract by hash.
     * 
     * @param hash contract hash
     * @return contract data
     * @throws BusinessException business exception
     */
    public JSONObject findByHash(String hash) throws BusinessException {
        return contractService.findByHash(hash);
    }

    /**
     * Find contract state by hash.
     * 
     * @param hash contract state hash
     * @return contract state data
     * @throws BusinessException business exception
     */
    public ContractState findStateByHash(String hash) throws BusinessException {
        return contractService.findStateByHash(hash);
    }

    /**
     * Find contract and state by contract hash.
     * 
     * @param hash contract hash
     * @return contract and state data
     * @throws BusinessException business exception
     */
    public JSONObject findExByHash(String hash) throws BusinessException {
        return contractService.findExByHash(hash);
    }

    /**
     * Find contract by name.
     * 
     * @param contractName contract name
     * @return contract data
     * @throws BusinessException business exception
     */
    public JSONObject findByName(String contractName) throws BusinessException {
        return contractService.findByName(contractName);
    }

    /**
     * Query contract history(up to 20 records).
     * 
     * @param contractName contract name
     * @return contract data
     * @throws BusinessException business exception
     */
    public List<JSONObject> queryContractHistory(String contractName)
        throws BusinessException {
        return contractService.queryContractHistory(contractName);
    }

    /**
     * Query contract state history(up to 20 records).
     * 
     * @param contractName contract name
     * @return contract state data
     * @throws BusinessException business exception
     */
    public List<JSONObject> queryContractStateHistory(String contractName)
        throws BusinessException {
        return contractService.queryContractStateHistory(contractName);
    }

    /**
     * Query contract attribute value from contract state.
     * 
     * @param csHash contract state hash(include history record)
     * @param attributeName attribute name in contract codes
     * @return attribute value
     * @throws BusinessException business exception
     */
    public String queryContractAttr(String csHash, String attributeName)
        throws BusinessException {
        if (StringUtils.isBlank(csHash) || StringUtils.isBlank(attributeName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (attributeName.contains(" ")) {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.SPACE_ILLEGAL, attributeName));
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put("contractTxHash", csHash);
        msg.getCommand().put("attributeName", attributeName);
        msg.setTargetType(RPCMessage.TargetType.QUERY_ATTR);
        return con.queryAndReturn(msg);
    }

    /**
     * Query contract attribute latest value from contract state.
     * 
     * @param contractName contract name
     * @param attributeName attribute name
     * @return attribute value
     * @throws BusinessException business exception
     */
    public String queryContractLatestAttr(String contractName,
                                          String attributeName)
        throws BusinessException {
        if (StringUtils.isBlank(contractName)
                || StringUtils.isBlank(attributeName)) {
            throw new BusinessException(ResultConstants.PARAM_ILLEGAL);
        }
        if (contractName.contains(" ")) {
            throw new BusinessException(ResultConstants
                    .getFailedMsg(ResultConstants.SPACE_ILLEGAL, contractName));
        }
        if (attributeName.contains(" ")) {
            throw new BusinessException(ResultConstants.getFailedMsg(
                    ResultConstants.SPACE_ILLEGAL, attributeName));
        }
        String hash = contractService.queryLatestCsHash(contractName);
        if (StringUtils.isBlank(hash)) {
            throw new BusinessException(ResultConstants.CONTRACT_NOT_EXISTS);
        }
        RPCMessage msg = con.getMessage();
        msg.getCommand().put("contractTxHash", hash);
        msg.getCommand().put("attributeName", attributeName);
        msg.setTargetType(RPCMessage.TargetType.QUERY_ATTR);
        return con.queryAndReturn(msg);

    }

}
