package cn.tdchain.cb.util;

import java.io.File;
import java.security.KeyStore;
import java.security.cert.X509Certificate;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.PropertiesConfiguration;

import cn.tdchain.cipher.Cipher;
import cn.tdchain.cipher.CipherException;
import cn.tdchain.cipher.Key;
import cn.tdchain.tdmsp.Msp;
import cn.tdchain.tdmsp.util.PkiUtil;

/**
 * System Configuration Utility.
 *
 * @version 1.0
 * @author bingoer.H 2018-11-27
 */
public class TdcbConfig {

    public static final int AIO = 1;
    public static final int NIO = 2;
    public static final int IO = 3;
    private static volatile TdcbConfig instance;
    private static PropertiesConfiguration config;
    private static final String SYSTEM_CONFIG_FILE = "tdcb.properties";

    public static final String TOKEN = "tdcb.connection.token";
    public static final String IPTABLE = "tdcb.connection.iptables";
    public static final String PORT = "tdcb.connection.port";
    public static final String CE_PORT = "tdcb.connection.ce.port";

    public static final String CERT_ROOT = "tdcb.cert.root";
    public static final String CERT_LOCAL = "tdcb.cert.local.keystore";
    public static final String CERT_LOCAL_PWD = "tdcb.cert.local.password";

    public static final String KS_PATH = "tdcb.contract.account.kspath";

    private String alias = "tdbc-key";

    private String connectionToken;
    private String[] ipTables;
    private int port;
    private int cePort;

    private String rootKspath;
    private String localKsPath;
    private String localKsPassword;

    private String accountKsPath;

    private Cipher cipher = new Cipher();
    private Key key = new Key();

    private void init() {

        String userJarDir = System.getProperty("user.dir");
        String filePath = userJarDir + "/" + SYSTEM_CONFIG_FILE;
        File file = new File(filePath);

        if (file.exists()) {
            try {
                config = new PropertiesConfiguration(filePath);
            } catch (ConfigurationException e) {
                throw new RuntimeException("Failed to load " + filePath + ".");
            }
        } else {
            try {
                config = new PropertiesConfiguration(SYSTEM_CONFIG_FILE);
            } catch (ConfigurationException e) {
                throw new RuntimeException(
                        "Failed to load " + SYSTEM_CONFIG_FILE + ".");
            }
        }

        // connection
        connectionToken = config.getString(TOKEN);
        String ips = config.getString(IPTABLE);
        if (ips != null && ips.length() > 0) {
            ipTables = ips.split(";");
        }

        port = config.getInt(PORT);
        cePort = config.getInt(CE_PORT);

        // cert
        if (config.containsKey(CERT_ROOT)) {
            rootKspath = config.getString(CERT_ROOT);
        }
        localKsPath = config.getString(CERT_LOCAL);
        localKsPassword = config.getString(CERT_LOCAL_PWD);
        if (StringUtils.isBlank(localKsPath)
                || StringUtils.isBlank(localKsPassword)) {
            throw new RuntimeException("Illegal keystore configuration.");
        }

        // contract
        accountKsPath = config.getString(KS_PATH);

        dealKey();

    }

    private void dealKey() {
        try {
            KeyStore keyStore = Msp.getKeyStore(localKsPath, localKsPassword);
            if (keyStore == null) {
                throw new CipherException("Local node KeyStore is null!");
            }

            String privateKey = cipher.getPrivateKeyStringByKeyStore(keyStore,
                    localKsPassword, Msp.ORGANIZATION_ALIAS);
            String publicKey = cipher.getPublicKeyStringByStore(keyStore,
                    localKsPassword, Msp.ORGANIZATION_ALIAS);

            if (StringUtils.isBlank(privateKey)
                    || StringUtils.isBlank(publicKey)) {
                throw new RuntimeException(
                        "Failed to load key pair: " + localKsPath + ".");
            }

            // 获取节点证书
            X509Certificate endCert = (X509Certificate) keyStore
                    .getCertificate(Msp.ORGANIZATION_ALIAS);
            String localCertBase64String = Msp.certToBase64String(endCert);

            key.setPrivateKey(privateKey);
            key.setPublicKey(publicKey);
            key.setLocalCertBase64String(localCertBase64String);

            if (!StringUtils.isBlank(rootKspath)) {
                // 获取根证书
                X509Certificate rootCert = PkiUtil.getCertFromCer(rootKspath);
                String rootCertBase64String = Msp.certToBase64String(rootCert);
                if (StringUtils.isBlank(rootCertBase64String)) {
                    throw new RuntimeException(
                            "Failed to load root cert: " + rootKspath + ".");
                }
                key.setRootCertBase64String(rootCertBase64String); // 缓存证书base64字符串
            }

        } catch (Exception e) {
            throw new CipherException(
                    "get private key by key store error:" + e.getMessage());
        }
    }

    /**
     * Get instance.
     * 
     * @return 单例模式 systemConig实例对象
     */
    public static synchronized TdcbConfig getInstance() {
        if (instance == null) {
            instance = new TdcbConfig();
            instance.init();
        }
        return instance;
    }

    /**
     * 应用用户用,进程使用usernewKsPath.
     * 
     * @return userKsPath
     */
    public String getAccountKsPath() {
        return accountKsPath;
    }

    public String getLocalKsPath() {
        return localKsPath;
    }

    public String getLocalKsPassword() {
        return localKsPassword;
    }

    public String getConnectionToken() {
        return connectionToken;
    }

    public String[] getIpTables() {
        return ipTables;
    }

    public int getPort() {
        return port;
    }

    public int getCePort() {
        return cePort;
    }

    public Cipher getCipher() {
        return cipher;
    }

    public Key getKey() {
        return key;
    }

    public String getAlias() {
        return alias;
    }

}
