package cn.tdchain.cb.service.impl;

import java.io.File;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Arrays;

import cn.tdchain.cb.constant.Commons;
import cn.tdchain.cb.constant.ResultConstants;
import cn.tdchain.cb.exception.BusinessException;
import cn.tdchain.cb.service.CipherService;
import cn.tdchain.cb.util.AddressUtils;
import cn.tdchain.cb.util.Base64Util;
import cn.tdchain.cb.util.StringUtils;
import cn.tdchain.cb.util.TdcbConfig;
import cn.tdchain.cipher.Cipher;

/**
 * Service Implementation for Cipher.
 *
 * @version 1.0
 * @author bingoer.H 2018-12-04
 */
public class CipherServiceImpl implements CipherService {

    private static final int PUBLIC_KEY_SIZE = 64;
    private static final int ADDRESS_SIZE = 160;
    private static final int ADDRESS_LENGTH_IN_HEX = ADDRESS_SIZE >> 2;
    private static final int PUBLIC_KEY_LENGTH_IN_HEX = PUBLIC_KEY_SIZE << 1;

    private String alias = TdcbConfig.getInstance().getAlias();
    private String[] cipherTypes = { Cipher.Type.RSA.name(),
        Cipher.Type.SM.name() };
    private Cipher rsaCipher = new Cipher();
    private Cipher smCipher = new Cipher();

    private Cipher getCipher(String crypto) {
        if (Cipher.Type.SM.name().equals(crypto)) {
            return smCipher;
        } else {
            return rsaCipher;
        }
    }

    @Override
    public String generateKeyStore(String crypto, String ksPassword)
        throws BusinessException {
        if (!Arrays.asList(cipherTypes).contains(crypto)) {
            throw new BusinessException(ResultConstants.CRYPTO_ERROR);
        }
        if (StringUtils.isBlank(ksPassword)) {
            ksPassword = Commons.DEFAULT_PWD;
        }

        String tempFilePath = AddressUtils.getTempFilePath();
        File file = new File(tempFilePath);
        if (file.exists()) {
            file.delete();
        }
        Cipher cipher = getCipher(crypto);
        // TODO: SM实现注掉了
        cipher.generateKeyStoreFile(tempFilePath, ksPassword, alias);

        LocalDateTime start = LocalDateTime.now();
        while (true) {
            if (file.exists()) {

                String publicKeyStr;
                try {
                    publicKeyStr = getPublicKey(cipher, tempFilePath,
                            ksPassword);
                } catch (Exception e) {
                    throw new BusinessException(
                            ResultConstants.PUBLIC_KEY_FAILED);
                }
                String address = getAddress(crypto, publicKeyStr);
                if (StringUtils.isBlank(address)) {
                    throw new BusinessException(ResultConstants.ADDRESS_FAILED);
                }
                boolean rename = file.renameTo(
                        new File(AddressUtils.getKsFilePath(address)));
                System.out.println("Create keystore:" + rename);
                return address;
            }
            long duration = Duration.between(start, LocalDateTime.now())
                    .toMillis();
            if (duration > 3000) {
                throw new BusinessException(ResultConstants.TIMEOUT);
            }
        }
    }

    @Override
    public String getAddress(String crypto, String publicKey) {
        if (StringUtils.isBlank(crypto) || StringUtils.isBlank(publicKey)) {
            return null;
        }
        if (!Arrays.asList(cipherTypes).contains(crypto)) {
            return null;
        }
        if (publicKey.length() < PUBLIC_KEY_LENGTH_IN_HEX) {
            publicKey = StringUtils.zeros(
                    PUBLIC_KEY_LENGTH_IN_HEX - publicKey.length()) + publicKey;
        }
        Cipher cipher = getCipher(crypto);
        String hash = cipher.hash(publicKey);
        if (hash == null) {
            return null;
        }
        return hash.substring(hash.length() - ADDRESS_LENGTH_IN_HEX); // right most 160 bits
    }

    @Override
    public String getPublicKeyStr(String crypto, String filePath,
                                  String ksPassword)
        throws BusinessException {
        if (StringUtils.isBlank(crypto) || StringUtils.isBlank(filePath)
                || StringUtils.isBlank(ksPassword)) {
            throw new BusinessException(ResultConstants.PUBLIC_KEY_FAILED);
        }
        if (!Arrays.asList(cipherTypes).contains(crypto)) {
            throw new BusinessException(ResultConstants.CRYPTO_ERROR);
        }
        Cipher cipher = getCipher(crypto);
        try {
            return getPublicKey(cipher, filePath, ksPassword);
        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(ResultConstants.PUBLIC_KEY_FAILED);
        }
    }

    @Override
    public String getPrivateKeyStr(String crypto, String filePath,
                                   String ksPassword) {
        if (StringUtils.isBlank(crypto) || StringUtils.isBlank(filePath)
                || StringUtils.isBlank(ksPassword)) {
            return null;
        }
        if (!Arrays.asList(cipherTypes).contains(crypto)) {
            return null;
        }
        Cipher cipher = getCipher(crypto);
        try {
            return getPrivateKey(cipher, filePath, ksPassword);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    @Override
    public String encrypt(String crypto, String data, String publicKeyStr)
        throws BusinessException {
        if (StringUtils.isBlank(crypto) || StringUtils.isBlank(data)
                || StringUtils.isBlank(publicKeyStr)) {
            throw new BusinessException(ResultConstants.USERINFO_EN_FAILED);
        }
        Cipher cipher = getCipher(crypto);
        return cipher.encryptByPublicKey(data, publicKeyStr);
    }

    @Override
    public String decrypt(String crypto, String data, String privateKeyStr) {
        if (StringUtils.isBlank(crypto) || StringUtils.isBlank(data)
                || StringUtils.isBlank(privateKeyStr)) {
            return data;
        }
        Cipher cipher = getCipher(crypto);
        return cipher.decryptByPrivateKey(data, privateKeyStr);
    }

    private String getPublicKey(Cipher cipher, String ksPath, String ksPwd)
        throws Exception {
        PublicKey publicKey = cipher.getPublicKeyByStore(ksPath, ksPwd, alias);

        return Base64Util.encoder(publicKey.getEncoded());
    }

    private String getPrivateKey(Cipher cipher, String ksPath, String ksPwd)
        throws Exception {
        PrivateKey privateKey = cipher.getPrivateKeyByKeyStore(ksPath, ksPwd,
                alias);
        return Base64Util.encoder(privateKey.getEncoded());
    }

}
