
package cn.tdchain.cb.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.serializer.SerializerFeature;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;

/**
 * Json Utility.
 *
 * @author Homer.J 2018-06-26
 * @version 1.0
 */
public final class JsonUtils {

    private JsonUtils() {
    }

    /**
     * Serializes the specified object into its equivalent Json representation.
     *
     * @param src specified object
     * @return Json String
     */
    public static String toJson(Object src) {
        return JSON.toJSONString(src);
    }

    /**
     * Serializes the specified object into its equivalent Json representation.
     *
     * @param src specified object
     * @param features SerializerFeatures
     * @return Json String
     */
    public static String toJson(Object src, SerializerFeature... features) {
        return JSON.toJSONString(src, features);
    }

    /**
     * Deserializes the specified Json into an object of the specified type.
     *
     * @param json Json String
     * @param typeOfT Type
     * @param <T> type
     * @return an object of the specified type, null if type not match
     */
    public static <T> T fromJson(String json, Type typeOfT) {
        try {
            return JSON.parseObject(json, typeOfT);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Get {@code List<String>}.
     *
     * @param json Json String
     * @return List
     */
    public static List<String> fromListString(String json) {
        try {
            return JSON.parseObject(json, TypeReference.LIST_STRING);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Get {@code Map<String, String>}.
     *
     * @param json Json String
     * @return Map
     */
    public static Map<String, String> fromMapString(String json) {
        try {
            return JSON.parseObject(json,
                    new TypeReference<Map<String, String>>() {
                    }.getType());
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Convert from bytes to object.
     *
     * @param bytes byte[]
     * @param classType Class
     * @return object
     */
    public static Object bytes2Object(byte[] bytes, Class<?> classType) {
        return JSON.parseObject(bytes, classType);
    }

    /**
     * Convert from object to bytes.
     *
     * @param object Object
     * @return bytes
     */
    public static byte[] object2Bytes(Object object) {
        return JSON.toJSONBytes(object);
    }
}
