package cn.tdchain.cb.util;

import java.io.File;
import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.List;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;

import cn.tdchain.cipher.Cipher;
import cn.tdchain.cipher.CipherException;
import cn.tdchain.cipher.Key;
import cn.tdchain.tdmsp.Msp;
import cn.tdchain.tdmsp.util.PkiUtil;

/**
 * System Configuration Utility.
 *
 * @version 1.0
 * @author bingoer.H 2018-11-27
 */
public class TdcbConfig {

    public static final int AIO = 1;
    public static final int NIO = 2;
    public static final int IO = 3;
    private static volatile TdcbConfig instance;
    private static final String SYSTEM_CONFIG_FILE = "tdcb.conf";
    private String alias = "tdbc-key";

    private String connectionToken;
    private String[] ipTables;
    private int port;
    private int cePort;

    private String rootKspath;
    private String localKsPath;
    private String localKsPassword;

    private String accountKsPath;

    private Cipher cipher = new Cipher();
    private Key key = new Key();

    private void init() {
        String userJarDir = System.getProperty("user.dir");
        String filePath = userJarDir + "/" + SYSTEM_CONFIG_FILE;
        File file = new File(filePath);
        Config config;
        if (file.exists()) {
            config = ConfigFactory.parseFile(file);
        } else {
            config = ConfigFactory.load(SYSTEM_CONFIG_FILE);
        }

        // connection
        connectionToken = config.getString("tdcb.connection.token");
        List<String> iplist = config.getStringList("tdcb.connection.iptables");
        ipTables = new String[iplist.size()];
        iplist.toArray(ipTables);
        port = config.getInt("tdcb.connection.port");
        cePort = config.getInt("tdcb.connection.ce.port");

        // cert
        if (config.hasPath("tdcb.cert.root")) {
            rootKspath = config.getString("tdcb.cert.root");
        }
        localKsPath = config.getString("tdcb.cert.local.keystore");
        localKsPassword = config.getString("tdcb.cert.local.password");
        if (StringUtils.isBlank(localKsPath)
                || StringUtils.isBlank(localKsPassword)) {
            throw new RuntimeException("Illegal keystore configuration.");
        }

        // contract
        accountKsPath = config.getString("tdcb.contract.account.kspath");

        dealKey();

    }

    private void dealKey() {
        try {
            KeyStore keyStore = Msp.getKeyStore(localKsPath, localKsPassword);
            if (keyStore == null) {
                throw new CipherException("Local node KeyStore is null!");
            }

            String privateKey = cipher.getPrivateKeyStringByKeyStore(keyStore,
                    localKsPassword, Msp.ORGANIZATION_ALIAS);
            String publicKey = cipher.getPublicKeyStringByStore(keyStore,
                    localKsPassword, Msp.ORGANIZATION_ALIAS);

            if (StringUtils.isBlank(privateKey)
                    || StringUtils.isBlank(publicKey)) {
                throw new RuntimeException(
                        "Failed to load key pair: " + localKsPath + ".");
            }

            // 获取节点证书
            X509Certificate endCert = (X509Certificate) keyStore
                    .getCertificate(Msp.ORGANIZATION_ALIAS);
            String localCertBase64String = Msp.certToBase64String(endCert);

            key.setPrivateKey(privateKey);
            key.setPublicKey(publicKey);
            key.setLocalCertBase64String(localCertBase64String);

            if (!StringUtils.isBlank(rootKspath)) {
                // 获取根证书
                X509Certificate rootCert = PkiUtil.getCertFromCer(rootKspath);
                String rootCertBase64String = Msp.certToBase64String(rootCert);
                if (StringUtils.isBlank(rootCertBase64String)) {
                    throw new RuntimeException(
                            "Failed to load root cert: " + rootKspath + ".");
                }
                key.setRootCertBase64String(rootCertBase64String); // 缓存证书base64字符串
            }

        } catch (Exception e) {
            throw new CipherException(
                    "get private key by key store error:" + e.getMessage());
        }
    }

    /**
     * Get instance.
     * 
     * @return 单例模式 systemConig实例对象
     */
    public static synchronized TdcbConfig getInstance() {
        if (instance == null) {
            instance = new TdcbConfig();
            instance.init();
        }
        return instance;
    }

    /**
     * 应用用户用,进程使用usernewKsPath.
     * 
     * @return userKsPath
     */
    public String getAccountKsPath() {
        return accountKsPath;
    }

    public String getLocalKsPath() {
        return localKsPath;
    }

    public String getLocalKsPassword() {
        return localKsPassword;
    }

    public String getConnectionToken() {
        return connectionToken;
    }

    public String[] getIpTables() {
        return ipTables;
    }

    public int getPort() {
        return port;
    }

    public int getCePort() {
        return cePort;
    }

    public Cipher getCipher() {
        return cipher;
    }

    public Key getKey() {
        return key;
    }

    public String getAlias() {
        return alias;
    }

}
