/*
 * Copyright 2025 Toint (599818663@qq.com)
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.toint.oktool.util;

import cn.hutool.v7.core.array.ArrayUtil;
import cn.hutool.v7.core.text.StrUtil;
import cn.hutool.v7.extra.validation.ValidationUtil;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.Contract;

import java.util.*;


/**
 * 断言工具
 *
 * @author Toint
 * @since 2025/5/28
 */
public class Assert {

    @Contract("null -> fail")
    public static void notNull(Object object) {
        if (object == null) {
            throw new IllegalArgumentException("参数不能为空");
        }
    }

    @Contract("null, _, _ -> fail")
    public static void notNull(Object object, CharSequence template, Object... params) {
        if (object == null) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("null, _ -> fail")
    public static void notNullParam(Object object, CharSequence paramName) {
        if (object == null) {
            throw new IllegalArgumentException(paramName + "不能为空");
        }
    }

    @Contract("!null -> fail")
    public static void isNull(Object object) {
        if (object != null) {
            throw new IllegalArgumentException("参数必须为空");
        }
    }

    @Contract("!null, _, _ -> fail")
    public static void isNull(Object object, CharSequence template, Object... params) {
        if (object != null) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("!null, _ -> fail")
    public static void isNullParam(Object object, CharSequence paramName) {
        if (object != null) {
            throw new IllegalArgumentException(paramName + "必须为空");
        }
    }

    @Contract("null -> fail")
    public static void notBlank(CharSequence text) {
        if (StringUtils.isBlank(text)) {
            throw new IllegalArgumentException("参数不能为空");
        }
    }

    @Contract("null, _, _ -> fail")
    public static void notBlank(CharSequence text, CharSequence template, Object... params) {
        if (StringUtils.isBlank(text)) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("null, _ -> fail")
    public static void notBlankParam(CharSequence text, CharSequence paramName) {
        if (StringUtils.isBlank(text)) {
            throw new IllegalArgumentException(paramName + "不能为空");
        }
    }

    @Contract("null -> fail")
    public static void notEmpty(Object[] arr) {
        if (arr == null || arr.length == 0) {
            throw new IllegalArgumentException("参数不能为空");
        }
    }

    @Contract("null, _, _ -> fail")
    public static void notEmpty(Object[] arr, CharSequence template, Object... params) {
        if (arr == null || arr.length == 0) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("null, _ -> fail")
    public static void notEmptyParam(Object[] arr, CharSequence paramName) {
        if (arr == null || arr.length == 0) {
            throw new IllegalArgumentException(paramName + "参数不能为空");
        }
    }

    @Contract("null -> fail")
    public static void notEmpty(Iterable<?> collection) {
        if (collection == null || !collection.iterator().hasNext()) {
            throw new IllegalArgumentException("参数不能为空");
        }
    }

    @Contract("null, _, _ -> fail")
    public static void notEmpty(Iterable<?> collection, CharSequence template, Object... params) {
        if (collection == null || !collection.iterator().hasNext()) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("null, _ -> fail")
    public static void notEmptyParam(Iterable<?> collection, CharSequence paramName) {
        if (collection == null || !collection.iterator().hasNext()) {
            throw new IllegalArgumentException(paramName + "不能为空");
        }
    }

    @Contract("null, _, _ -> fail")
    public static void notEmpty(Map<?, ?> map, CharSequence template, Object... params) {
        if (map == null || map.isEmpty()) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("null, _ -> fail")
    public static void notEmptyParam(Map<?, ?> map, CharSequence paramName) {
        if (map == null || map.isEmpty()) {
            throw new IllegalArgumentException(paramName + "不能为空");
        }
    }

    public static void equals(Object a, Object b, CharSequence template, Object... params) {
        if (!Objects.equals(a, b)) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    public static void notEquals(Object a, Object b, CharSequence template, Object... params) {
        if (Objects.equals(a, b)) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("false, _, _ -> fail")
    public static void isTrue(boolean b, CharSequence template, Object... params) {
        if (!b) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("true, _, _ -> fail")
    public static void isFalse(boolean b, CharSequence template, Object... params) {
        if (b) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }
    }

    @Contract("null, _ -> fail")
    public static void validate(Object object, Class<?>... groups) {
        Assert.notNull(object);
        ValidationUtil.validateAndThrowFirst(object, groups);
    }

    /**
     * 验证对象, 失败则抛异常
     *
     * <p>如果校验失败, 异常信息会添加到 {@code params} 数组末尾, 调用者可在 {@code template} 预留位置, 否则忽略</p>
     */
    @Contract("null, _, _ -> fail")
    public static void validate(Object object, CharSequence template, Object... params) {
        if (object == null) {
            throw new IllegalArgumentException(StrUtil.format(template, params));
        }

        try {
            ValidationUtil.validateAndThrowFirst(object);
        } catch (Exception e) {
            if (StringUtils.isBlank(template)) {
                throw new IllegalArgumentException(e.getMessage(), e);
            }

            List<Object> newParams = new ArrayList<>();
            if (ArrayUtil.isNotEmpty(params)) {
                newParams.addAll(Arrays.asList(params));
            }
            newParams.add(e.getMessage());
            throw new IllegalArgumentException(StrUtil.format(template, newParams.toArray()), e);
        }
    }
}