package cn.tworice.backup.service;

import cn.tworice.backup.executor.BackupExecutor;
import cn.tworice.backup.executor.DatabaseEnum;
import cn.tworice.backup.executor.DatabaseParam;
import cn.tworice.backup.scheduler.BackupScheduledTask;
import cn.tworice.common.framework.mail.bo.Attachment;
import cn.tworice.common.framework.mail.bo.SendMailBO;
import cn.tworice.common.framework.mail.core.MailExecutor;
import cn.tworice.common.util.StringUtils;
import cn.tworice.common.util.spring.ApplicationUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 备份服务
 */
@Service
@Slf4j
public class BackupService {

    @Value("${tworice.backup.path:''}")
    private String path;

    @Resource
    private MailExecutor mailExecutor;

    @Resource
    private DataSourceProperties dataSourceProperties;

    @Value("${tworice.backup.email:''}")
    private String email;

    @Value("${tworice.backup.cron:0 0 2 * * *}")
    private String cron;

    @Value("${tworice.backup.enabled:false}")
    private Boolean enabled;


    @Resource
    private BackupScheduledTask backupScheduledTask;

    @PostConstruct
    public void init() {
        if(!enabled){
            return;
        }
        DatabaseParam param = new DatabaseParam();
        param.setDatabaseName(this.getDatabaseName(dataSourceProperties.getUrl()));
        param.setType(this.getType(dataSourceProperties.getUrl()));
        param.setUrl(dataSourceProperties.getUrl());
        param.setDriver(dataSourceProperties.getDriverClassName());
        param.setUsername(dataSourceProperties.getUsername());
        param.setPassword(dataSourceProperties.getPassword());
        param.setEmail(email);

        log.info("初始化数据库备份计划：{},周期：{}",param,cron);
        this.scheduleTaskAtSpecificTime(param, cron);
    }

    public void scheduleTaskAtSpecificTime(DatabaseParam param,int hour, int minute) {
        Trigger trigger = new CronTrigger(String.format("0 %d %d * * *", minute, hour));
        backupScheduledTask.scheduleTask(param,trigger);
    }

    public void scheduleTaskAtSpecificTime(DatabaseParam param,String cron) {
        Trigger trigger = new CronTrigger(cron);
        backupScheduledTask.scheduleTask(param, trigger);
    }

    public void backup(DatabaseParam param) {
        BackupExecutor executor= this.getBackupExecutorByType(param);
        if (StringUtils.isBlank(path)) {
            path = System.getProperty("user.dir") + File.separator + "backup";
        }else{
            path = System.getProperty("user.dir") + File.separator + path;
        }
        StringBuilder builder = new StringBuilder(path);
        executor.backup(builder);
        log.info("数据库备份执行成功：{}",builder);
        this.sendMail(builder.toString(),param);
    }

    public BackupExecutor getBackupExecutorByType(DatabaseParam param) {
        DatabaseEnum executorByType = DatabaseEnum.getExecutorByType(param.getType());
        if (executorByType == null) {
            throw new RuntimeException("未找到数据库类型");
        }
        try {
            return executorByType.getClazz().getConstructor(DatabaseParam.class).newInstance(param);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void sendMail(String path,DatabaseParam param) {
        if (StringUtils.isBlank(param.getEmail()) || StringUtils.isBlank(path)) {
            return;
        }
        SendMailBO sendMailBO = new SendMailBO();
        sendMailBO.setTo(param.getEmail());
        sendMailBO.setSubject("【数据库备份】"+param.getDatabaseName());
        sendMailBO.setSimple(false);
        sendMailBO.setContent("数据库备份");

        File file = new File(path);
        Attachment  attachment= new Attachment();
        attachment.setName(file.getName());
        attachment.setFile(file);
        sendMailBO.setAttachment(Collections.singletonList(attachment));
        mailExecutor.sendMail(sendMailBO);
    }

    private String getDatabaseName(String url) {
        String[] urlParts = url.split("/");
        return urlParts[urlParts.length - 1].split("\\?")[0];
    }

    private String getType(String url) {
        String[] urlParts = url.split(":");
        return urlParts[1];
    }
}
