package cn.tworice.generate.web;


import cn.tworice.common.util.StringUtils;
import cn.tworice.common.vo.RequestResult;
import cn.tworice.common.vo.StateCodeConst;
import cn.tworice.generate.constant.DataTypeConst;
import cn.tworice.generate.dao.CodeGenerationDao;
import cn.tworice.generate.exception.GenerateException;
import cn.tworice.generate.po.DataBaseTable;
import cn.tworice.generate.service.CodeGenerationService;
import cn.tworice.generate.service.VueGeneration;
import cn.tworice.generate.util.CodeUtils;
import cn.tworice.generate.vo.CodeVO;
import cn.tworice.generate.vo.CreateServerVO;
import cn.tworice.generate.vo.Parameter;
import cn.tworice.log.annotation.Log;
import com.alibaba.fastjson.JSON;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * 代码生成
 *
 * @author 二饭[tworice@sina.cn][2022/1/8 20:54]
 */
@RestController
@RequestMapping("admin/codeGeneration")
@CrossOrigin
@Slf4j
@Api(tags = "代码生成")
@Log(ignore = true)
public class CodeGeneration {

    /**
     * Vue项目构建
     */
    @Resource
    private VueGeneration vueGeneration;

    /**
     * 后端代码生成
     */
    @Resource
    private CodeGenerationService codeGenerationService;

    /**
     * 通用的代码工具
     */
    @Resource
    private CodeUtils codeUtil;

    /**
     * 数据库操作
     */
    @Resource
    private CodeGenerationDao dao;

    /**
     * 获取非系统表
     *
     * @return cn.tworice.common.vo.RequestResult
     * @author 二饭 [2022/11/10]
     **/
    @GetMapping("tables")
    @ApiOperation("获取非系统表")
    public RequestResult showTables() {
        RequestResult result = new RequestResult();
        List<DataBaseTable> dataBaseTables = dao.selectTableList();
        for (int i = 0; i < dataBaseTables.size(); i++) {
            DataBaseTable dataBaseTable = dataBaseTables.get(i);
            if (dataBaseTable.getTableName().startsWith("system")) {
                dataBaseTables.remove(i);
                i--;
            } else {
                dataBaseTable.setName(dataBaseTable.getTableComment());
                dataBaseTable.setId(dataBaseTable.getTableName());
            }
        }
        result.appendData("list", dataBaseTables);
        return result;
    }

    /**
     * 通过模板名称获取模板内容
     *
     * @param templateName 模板名称
     * @return cn.tworice.common.vo.RequestResult
     * @author 二饭 [2022/11/10]
     **/
    @GetMapping("read")
    @ApiOperation("获取模板内容")
    public RequestResult readFile(String templateName) {
        RequestResult result = new RequestResult();
        String content = codeUtil.readByName(templateName);
        result.appendData("content", content);
        return result;
    }

    /**
     * 生成Vue代码
     *
     * @param codeVO 模板填充内容
     * @return cn.tworice.common.vo.RequestResult
     * @author 二饭 [2022/11/10]
     **/
    @PostMapping("writeVue")
    @ApiOperation("生成Vue代码")
    public RequestResult write(CodeVO codeVO) {
        if (vueGeneration.buildCodeVue(codeVO)) {
            return RequestResult.success(StateCodeConst.MESSAGE, "构建Vue页面成功");
        } else {
            return RequestResult.error("构建Vue页面失败");
        }
    }

    /**
     * 生成服务端及SQL代码
     *
     * @param createServerVO 参数
     *                       创建服务端页面及数据库
     * @author 二饭[tworice@sina.cn][2022/1/14 22:30]
     */
    @PostMapping("writeServe")
    @ApiOperation("生成服务端及SQL代码")
    public RequestResult write(CreateServerVO createServerVO) {
        List<Parameter> list = JSON.parseArray(createServerVO.getParameters(), Parameter.class);
        // 删除ID字段，默认id字段为自增
        for (int i = 0; i < list.size(); i++) {
            Parameter parameter = list.get(i);
            if (parameter.getField().equals("id")) {
                list.remove(i);
                i--;
            } else if (parameter.getQuery() && parameter.getQueryType().equals("数据表")) {
                String dictTable = parameter.getDict();
            }
        }
        try {
            dao.createNewTable(list, createServerVO.getTableName(), createServerVO.getTableComment());// 创建表
            log.info("创建数据表成功");
        } catch (Exception exception) {
            throw new GenerateException("数据表创建失败");
        }

        codeUtil.updateType(list);// 更新字段类型
        createServerVO.setFieldList(list);

        // 将表名进行驼峰命名转换
        createServerVO.setEntityName(StringUtils.lineToHump2(createServerVO.getTableName()));
        // 创建实体类
        boolean success = codeGenerationService.createEntity(createServerVO);
        // 创建Dao层
        success &= codeGenerationService.createDao(createServerVO);
        // 创建service文件
        success &= codeGenerationService.createService(createServerVO);
        // 创建Client文件
        success &= codeGenerationService.createClient(createServerVO);
        if (success) {
            return RequestResult.success(StateCodeConst.MESSAGE, "生成服务端文件成功");
        } else {
            return RequestResult.error("生成服务端文件失败");
        }
    }

    /**
     * 获取路由文件
     *
     * @return cn.tworice.common.vo.RequestResult
     * @author 二饭 [2022/11/10]
     **/
    @GetMapping("router")
    @ApiOperation("获取路由文件")
    public RequestResult getRouter() {
        return RequestResult.success().appendData("router", vueGeneration.getRouter());
    }

    @GetMapping("dbTypes")
    @ApiOperation("获取支持的数据库字段类型")
    public RequestResult dbTypes() {
        return RequestResult.success().appendData("dbTypes", DataTypeConst.getDbTypes());
    }
}
