package cn.tworice.monitor.service;

import cn.tworice.common.util.AgingMap;
import cn.tworice.ip.util.IpAddrUtils;
import cn.tworice.monitor.vo.RequestInfo;
import com.alibaba.fastjson.JSON;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.regex.Pattern;

@Service
public class RequestFlowService {
    @Getter private final ConcurrentLinkedQueue<RequestInfo> queue = new ConcurrentLinkedQueue<>();

    @Value("${tworice.monitor.flow:false}")
    private Boolean flow;

    @Value("${tworice.monitor.storage:300}")
    private Integer STORAGE;

    @Value("${tworice.monitor.LIMIT:200}")
    private Integer LIMIT;

    @Value("${tworice.monitor.refuse:60000}")
    private Long REFUSE;

    /**
     * 记录存在sql注入的记录
     */
    @Getter private final Map<String, String> sqlMap = new HashMap<>();

    /**
     * 被禁的Map集合
     */
    private final AgingMap<String, String> banMap = new AgingMap<>();

    /**
     * 判断请求频率
     * @param request 请求
     * @return 是否通过
     */
    public boolean enter(HttpServletRequest request) {
        if (!flow) {
            return true;
        }
        RequestInfo requestInfo = new RequestInfo();
        requestInfo.setIpAddr(IpAddrUtils.getIpAddr(request));
        if (queue.size() > this.STORAGE) {
            queue.poll();
        }
        requestInfo.setDate(format());
        queue.add(requestInfo);
        return record(requestInfo);
    }

    public boolean sqlInject(HttpServletRequest request) {
        // 获取请求路径
        String requestPath = request.getRequestURI();

        // 检查请求路径是否包含SQL注入
        if (SQL_INJECTION_PATTERN.matcher(requestPath).find()) {
            return true;
        }

        // 获取请求参数并检查是否包含SQL注入
        for (String[] paramValues : request.getParameterMap().values()) {
            for (String paramValue : paramValues) {
                if (paramValue != null && SQL_INJECTION_PATTERN.matcher(paramValue).find()) {
                    String ipAddr = IpAddrUtils.getIpAddr(request);
                    this.sqlMap.put(ipAddr, paramValue);
                    this.banMap.put(ipAddr, null, this.REFUSE);
                    return true;
                }
            }
        }

        // 如果没有发现SQL注入，返回false
        return false;
    }


    private boolean record(RequestInfo requestInfo) {
        if (this.banMap.containsKey(requestInfo.getIpAddr())) {
            return false;
        }
        String s = JSON.toJSONString(this.queue);
        String[] split = s.split(requestInfo.getIpAddr());
        if (split.length > this.LIMIT) {
            banMap.put(requestInfo.getIpAddr(), requestInfo.getDate(), this.REFUSE);
            return false;
        }
        return true;
    }

    private String format() {
        GregorianCalendar calendar = new GregorianCalendar();
        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        int minute = calendar.get(Calendar.MINUTE);
        return hour + ":" + minute;
    }

    /**
     * 定义一些常见的SQL注入模式
     */
    private static final Pattern SQL_INJECTION_PATTERN = Pattern.compile(
            "(?:\\b(?:select|update|delete|insert|truncate|alter|drop|create|exec|execute|declare|union|merge|call)\\b|\\b(?:--|;|'|\"|\\/\\*|\\*\\/|\\\\x|--|\\/\\*|\\*\\/|\\\\x).*)",
            Pattern.CASE_INSENSITIVE | Pattern.MULTILINE | Pattern.DOTALL
    );
}
