package cn.tworice.system.controller.role;

import cn.tworice.common.vo.RequestResult;
import cn.tworice.log.annotation.Log;
import cn.tworice.mybatis.vo.PageParam;
import cn.tworice.system.dao.role.po.RoleDO;
import cn.tworice.system.service.role.RoleService;
import cn.tworice.system.service.user.UserService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Arrays;


/**
 * 角色管理
 */
@RestController
@RequestMapping("admin/system/role")
@Slf4j
@CrossOrigin
@Api(tags = "系统角色")
@Log(table = "system_role")
public class RoleController {

    @Resource
    private RoleService roleService;

    @Resource
    private UserService userService;

    @GetMapping("list")
    @ApiOperation("获取角色列表")
    public RequestResult roleList(PageParam pageParam, RoleDO entity) {
        return RequestResult.success().setData(roleService.getRoleList(pageParam, entity));
    }

    @GetMapping("adminList")
    @ApiOperation("通过角色ID获取人员列表")
    public RequestResult adminList(PageParam pageParam, Integer roleId) {
        return RequestResult
                .success()
                .setData(userService.getUsersByRole(pageParam, roleId));
    }

    @PutMapping("adminToRole")
    @ApiOperation("添加人员到指定角色，可多人员")
    public RequestResult addAdminToRole(String[] userId, Integer roleId) {
        if (roleService.addUserToRole(userId, roleId)) {
            return RequestResult.success(201,"操作成功");
        } else {
            return RequestResult.error(304, "请勿重复添加人员到角色");
        }
    }

    @DeleteMapping("rmAdminToRole")
    @ApiOperation("从角色移除人员")
    public RequestResult rmAdminToRole(String[] adminIds, Integer roleId) {
        roleService.delAdminToRole(adminIds, roleId);
        return RequestResult.success(201,"操作成功");
    }

    @GetMapping("getNoRoleAdmin")
    @ApiOperation("获取不是该角色的人员")
    public RequestResult getNoRoleAdmin(Integer roleId) {
        return RequestResult.success().appendData("result", roleService.getNoRoleAdmin(roleId));
    }

    @GetMapping("getAdminByStatus")
    @ApiOperation("根据账号状态获取人员列表，1-正常，-1-非正常")
    public RequestResult getAdminAll(Integer status) {
        return RequestResult.success()
                .appendData("result", userService.getUsersByState(status));
    }

    @GetMapping("getResourcesByRoleId")
    @ApiOperation("根据角色ID查询资源 关联数据")
    public RequestResult getResourcesByRoleId(Integer roleId) {
        return RequestResult.success()
                .appendData("result", roleService.getResourcesByRoleId(roleId));
    }

    /**
     * 获取角色权限列表 暂无更新
     *
     * @author 二饭
     */
    @GetMapping("resourcesByRole")
    @ApiOperation("根据角色ID查询资源 关联数据")
    public RequestResult resourcesByRole(Integer roleId) {
        return RequestResult.success()
                .appendData("result", roleService.getResourcesByRoleId(roleId));
    }

    /**
     * @author 二饭
     */
    @PutMapping("updateRoleResources")
    @ApiOperation("更新角色权限")
    @Deprecated
    public RequestResult updateRoleResources(@ApiParam("原有资源ID数组") Integer[] haveResources, @ApiParam("更新后的资源ID数组") Integer[] updateResources, Integer roleId) {
        if (roleService.updateRoleResources(haveResources, updateResources, roleId)) {
            RequestResult.success(201,"更新成功");
        }
        return RequestResult.error();
    }

    /**
     * 更新角色权限 叶子节点版
     * @author 二饭
     */
    @PutMapping("updateResources")
    @ApiOperation("更新角色权限")
    public RequestResult updateResources(@ApiParam("角色ID") Integer roleId, @ApiParam("更新后的资源ID数组") Integer[] resourcesId) {
        roleService.setRoleResources(roleId, resourcesId);
        return RequestResult.success(201,"更新成功");
    }

    @DeleteMapping("delRole")
    @ApiOperation("删除角色")
    public RequestResult delRole(Integer[] roleId) {
        if (roleService.delRole(Arrays.asList(roleId))) {
            return RequestResult.success(201,"删除成功");
        }
        return RequestResult.error();
    }

    @PutMapping("editRole")
    @ApiOperation("编辑角色")
    public RequestResult editRole(Integer roleId, String roleName) {
        if (roleService.editRole(roleId, roleName) == 1) {
            return RequestResult.success(201,"保存成功");
        }
        return RequestResult.error();
    }
}
