package cn.tworice.system.controller.department;

import cn.tworice.common.vo.RequestResult;
import cn.tworice.log.annotation.Log;
import cn.tworice.system.dao.department.po.DepartmentDO;
import cn.tworice.system.service.department.DepartmentService;
import cn.tworice.system.service.user.UserService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Arrays;


/**
 * 部门管理
 */
@RestController
@RequestMapping("admin/system/department")
@Slf4j
@CrossOrigin
@Api(tags = "系统部门")
@Log(table = "system_department")
public class DepartmentController {

    @Autowired
    private DepartmentService departmentService;

    @Resource
    private UserService userService;

    @GetMapping("list")
    @ApiOperation("获取部门列表")
    public RequestResult departmentList() {
        return RequestResult.success()
                .appendData("departmentList", departmentService.list());
    }

    @GetMapping("adminList")
    @ApiOperation("通过部门ID获取人员列表")
    public RequestResult adminList(Integer departmentId) {
        return RequestResult.success()
                .appendData("adminList", userService.getUsersByDepartmentId(departmentId));
    }

    @PutMapping("adminToDepartment")
    @ApiOperation("添加人员到指定部门")
    public RequestResult addAdminToDepartment(String[] userId, Integer departmentId) {
        if (departmentService.addAdminToDepartment(userId, departmentId)) {
            return RequestResult.success("操作成功");
        } else {
            return RequestResult.error( "请勿重复添加人员到部门");
        }
    }

    @DeleteMapping("rmAdminToDepartment")
    @ApiOperation("从部门移除人员")
    public RequestResult rmAdminToDepartment(String[] adminIds, Integer departmentId) {
        departmentService.delAdminToDepartment(adminIds, departmentId);
        return RequestResult.success("操作成功");
    }

    @GetMapping("getNoDepartmentAdmin")
    @ApiOperation("获取不是该部门的人员")
    public RequestResult getNoDepartmentAdmin(Integer departmentId) {
        return RequestResult.success().appendData("result", userService
                .getUsersByNoDepartmentId(departmentId));
    }

    @GetMapping("getAdminByStatus")
    @ApiOperation("根据账号状态获取人员列表，0-正常，1-非正常")
    public RequestResult getAdminAll(Integer status) {
        return RequestResult.success()
                .appendData("result", userService.getUsersByState(status));
    }

    @DeleteMapping("delDepartment")
    @ApiOperation("删除部门")
    public RequestResult delDepartment(Integer[] departmentId) {
        boolean b = departmentService.delDepartment(Arrays.asList(departmentId));
        if (b) {
            return RequestResult.success("删除成功")
                    .appendData("departmentList", departmentService.list());
        }else{
            return RequestResult.error();
        }
    }

    @PutMapping("editDepartment")
    @ApiOperation("编辑部门")
    public RequestResult editDepartment(Integer departmentId, String departmentName) {
        if (departmentId == null || departmentId < 1) {
            departmentService.save(new DepartmentDO().setDepartmentName(departmentName));
        }else {
            departmentService.updateById(new DepartmentDO().setId(departmentId).setDepartmentName(departmentName));
        }
        return RequestResult.success("保存成功")
                .appendData("departmentList", departmentService.list());
    }
}
