package cn.tworice.system.controller.login;

import cn.tworice.auth.service.AuthManager;
import cn.tworice.captcha.core.CaptchaBase64;
import cn.tworice.common.annotation.NoEncryption;
import cn.tworice.log.annotation.Log;
import cn.tworice.system.controller.login.vo.RegVO;
import cn.tworice.common.vo.RequestResult;
import cn.tworice.system.service.login.LoginService;
import cn.tworice.system.controller.login.vo.LoginVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 登录注册操作及登录页数据展示
 *
 * @author 二饭
 */
@RestController
@RequestMapping("admin/login")
@Slf4j
@CrossOrigin
@Api(tags = "系统登录")
@Log(ignore = true)
public class LoginController {

    @Resource
    private LoginService loginService;

    @Resource
    private AuthManager authManager;

    @ApiOperation("登录")
    @PostMapping()
    public RequestResult login(LoginVO loginVO, HttpServletRequest request, HttpServletResponse response) {
        if(authManager.auth(request, response)){
            return loginService.checkLogin(loginVO, request);
        }
        return RequestResult.error("登录失败");
    }

    @ApiOperation("邮箱验证码登录")
    @PostMapping("emailLogin")
    public RequestResult emailLogin(LoginVO loginVO, HttpServletRequest request, HttpServletResponse response) {
        if(authManager.auth(request, response)){
            return loginService.checkMailCaptchaLogin(loginVO, request);
        }
        return RequestResult.error("登录失败");
    }

    @ApiOperation("获取邮箱验证码")
    @PostMapping("emailCaptcha")
    public RequestResult emailCaptcha(String email) {
        loginService.getCaptcha(email);
        return RequestResult.success();
    }

    @ApiOperation("二次验证登录")
    @PostMapping("reAuth")
    public RequestResult login(String key,String varCode, HttpServletRequest request) {
        return loginService.checkLogin(key, varCode,request);
    }

    @ApiOperation("重新发送二次登录验证码")
    @PostMapping("sendReAuthCaptcha")
    public RequestResult sendReAuthCaptcha(String key) {
        loginService.sendReAuthCaptcha(key);
        return RequestResult.success();
    }

    /**
     * 2022年5月13日 将原Redis替换为AgingMap，并添加切换验证码时使原验证码失效
     * @param code 用户切换验证码时删除原验证码
     * @author 二饭
     */
    @GetMapping("captcha")
    @ApiOperation("获取验证码图片")
    @NoEncryption
    public RequestResult captcha(String code){
        CaptchaBase64 captcha = loginService.getLoginCaptcha(code);
        // 构建返回客户端数据对象
        return RequestResult.success()
                .appendData("captcha", captcha.getCaptcha())
                .appendData("key", captcha.getKey());
    }

    /**
     * 发送邮箱验证码
     * @author 二饭[tworice@sina.cn]
     */
    @GetMapping("regCaptcha")
    @ApiOperation("发送邮箱验证码")
    public RequestResult regCaptcha(String email){
        return RequestResult.success("验证码已发送")
                .appendData("key", loginService.getRegCaptcha(email));
    }

    /**
     * 注册账号信息
     *
     * @param regVO 注册信息
     * @return cn.tworice.common.vo.RequestResult
     * @author 二饭
     **/
    @PostMapping("reg")
    @ApiOperation("注册账号信息")
    public RequestResult reg(RegVO regVO) {
        return RequestResult.success(loginService.checkReg(regVO));
    }
}
