package cn.wangzq.flog.service;

import android.annotation.SuppressLint;
import android.system.Os;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.math.BigInteger;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Locale;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import cn.wangzq.flog.ILog;
import cn.wangzq.flog.Level;

public class FileLog implements ILog {

    public static final String TYPE = "file";

    private static final String KEY_FILE_FOUNT = TYPE + ".max_file_count";
    private static final String KEY_FILE_SIZE = TYPE + ".max_file_size";
    private static final String KEY_FILE_PATH = TYPE + ".log_path";
    private static final String KEY_FILE_PREFIX = TYPE + ".file_prefix";
    private static final String KEY_GZ_ENABLE = TYPE + ".gz_enable";
    private static final String KEY_LOG_LEVEL = TYPE + ".level";

    private int maxFileCount = 10;
    private long maxFileSize = 1024 * 1024 * 5;
    private String logPath = "flog";
    private String filePrefix = "flog";
    private boolean gzEnable = false;
    private int mLevel = Level.VERBOSE;

    private final ExecutorService mZipThread = new ThreadPoolExecutor(1, 1,
            0L, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(),
            r -> new Thread(r, "FLOG_ZIP"));

    private final ExecutorService mWriteThread = new ThreadPoolExecutor(1, 100,
            0L, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(),
            new ThreadFactory() {
                private final AtomicInteger threadNumber = new AtomicInteger(1);
                @Override
                public Thread newThread(Runnable r) {
                    SecurityManager s = System.getSecurityManager();
                    ThreadGroup group;
                    group = (s != null) ? s.getThreadGroup() :
                            Thread.currentThread().getThreadGroup();
                    Thread t = new Thread(group, r,
                            "FLOG_WRITE:" + threadNumber.getAndIncrement(),
                            0);
                    if (t.isDaemon())
                        t.setDaemon(false);
                    if (t.getPriority() != Thread.NORM_PRIORITY)
                        t.setPriority(Thread.NORM_PRIORITY);
                    return t;
                }
            });
    public FileLog() {
    }

    @SuppressLint("SimpleDateFormat")
    private final DateFormat mTimeFormat = new SimpleDateFormat("Z yyyy-MM-dd HH:mm:ss.SSS");

    private String getSourceInfo() {
        StackTraceElement stack = new Throwable().getStackTrace()[4];
        return stack.getFileName() + ":" + stack.getLineNumber();
    }

    private void sendWriteMessage(String level, String tag, String msg) {
        String time = mTimeFormat.format(Calendar.getInstance().getTime());
        int pid = Os.getpid();
        int tid = Os.gettid();
        String source = getSourceInfo();
        String str = String.format(Locale.getDefault(),
                "[%s] %s [%d-%d] %s: %s @ %s\n",
                level, time, pid, tid, tag, msg, source);
        mWriteThread.execute(() -> writeToFile(str));
    }

    private void writeToFile(String str) {
        try (FileOutputStream fos = new FileOutputStream(getLogFile(), true);
             BufferedOutputStream out = new BufferedOutputStream(fos)) {
            out.write(str.getBytes());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private synchronized void zipLogs() {
        try {
            if (!gzEnable) return;
            File[] allLog = new File(logPath).listFiles(LOG_FILTER);
            if (allLog == null || allLog.length <= 1) {
                return;
            }
            sortLogFileList(allLog);

            for (int i = 0; i < allLog.length - 1; i++) {
                File source = allLog[i];
                if (!source.exists()) continue;
                String name = source.getName();
                if (!name.endsWith(".log")) continue;

                File target = new File(source.getParentFile(), name.replace(".log", ".zip"));
                try (FileInputStream inputStream = new FileInputStream(source);
                     ZipOutputStream o = new ZipOutputStream(new FileOutputStream(target));) {
                    ZipEntry entry = new ZipEntry(source.getName());
                    o.putNextEntry(entry);
                    int len;
                    byte[] buffer = new byte[4096];
                    while ((len = inputStream.read(buffer)) != -1) {
                        o.write(buffer, 0, len);
                    }
                    o.flush();
                } catch (Exception e) {
                    w("zip file error!", e);
                }
                source.delete();
            }
        } finally {
            removeOldLogs();
        }
    }


    private boolean isOversize(File file) {
        if (null == file) return false;
        boolean res = file.length() > maxFileSize;
        if (res) {
            mZipThread.execute(this::zipLogs);
        }
        return res;
    }


    private String getFormatFileName(BigInteger index) {
        return String.format(Locale.getDefault(), "%s.%04d.log", filePrefix, index);
    }

    private synchronized void removeOldLogs() {
        File[] allLog = new File(logPath).listFiles(LOG_FILTER);
        if (allLog == null || allLog.length <= maxFileCount) {
            return;
        }
        sortLogFileList(allLog);
        for (int i = 0; i < allLog.length - maxFileCount; i++) {
            //noinspection ResultOfMethodCallIgnored
            allLog[i].delete();
        }

    }


    private static File sLastFile;

    private File getLastLogFile(String path, BigInteger index) {

        File logFile = new File(path, getFormatFileName(index));
        if (logFile.exists() && isOversize(logFile)) {
            index = index.add(new BigInteger("1"));
            logFile = getLastLogFile(path, index);
        }
        return logFile;
    }

    private File getLogFile() {
        File dir = new File(logPath);
        if (!dir.exists()) {
            boolean res = dir.mkdirs();
        }

        if (null == sLastFile || !sLastFile.exists() || isOversize(sLastFile)) {
            File[] oldLogs = dir.listFiles(LOG_FILTER);
            String last = "0";
            if (oldLogs != null && oldLogs.length > 0) {
                sortLogFileList(oldLogs);
                String fileName = oldLogs[oldLogs.length - 1].getName();
                String[] sp = fileName.split("\\.");
                last = sp[sp.length - 2];
            }
            BigInteger index = new BigInteger(last);
            sLastFile = getLastLogFile(logPath, index);
        }
        return sLastFile;
    }

    private final FileFilter LOG_FILTER = pathname -> {
        String fileName = pathname.getName();
        boolean res = fileName.startsWith(filePrefix);
        res = res && (fileName.endsWith(".log") || fileName.endsWith(".zip"));
        return res;
    };

    private void sortLogFileList(File[] logs) {
        Arrays.sort(logs, (o1, o2) -> {
            int res = 0;
            try {
                String[] sp1 = o1.getName().split("\\.");
                String[] sp2 = o2.getName().split("\\.");
                String index1 = sp1[sp1.length - 2];
                String index2 = sp2[sp2.length - 2];
                res = new BigInteger(index1).subtract(new BigInteger(index2)).intValue();
            } catch (Exception e) {
                e.printStackTrace();
            }
            return res;
        });
    }

    private static String getStackTrace(Throwable tr) {
        if (tr == null) {
            return "";
        }
        Throwable t = tr;
        int count = 2;
        while (t != null && count > 0) {
            t = t.getCause();
            count--;
        }
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        tr.printStackTrace(pw);
        pw.flush();
        return sw.toString();
    }


    @Override
    public String getType() {
        return TYPE;
    }

    @Override
    public void load(Properties p) {
        maxFileCount = Integer.parseInt(p.getProperty(KEY_FILE_FOUNT, maxFileCount + ""));
        maxFileSize = Long.parseLong(p.getProperty(KEY_FILE_SIZE, maxFileSize + ""));
        logPath = p.getProperty(KEY_FILE_PATH, logPath);
        filePrefix = p.getProperty(KEY_FILE_PREFIX, filePrefix);
        gzEnable = Boolean.parseBoolean(p.getProperty(KEY_GZ_ENABLE));
        mLevel = Level.parse(p.getProperty(KEY_LOG_LEVEL, "V"));
    }

    @Override
    public void v(String tag, String msg) {
        sendWriteMessage("V", tag, msg);
    }

    @Override
    public void v(String tag, String msg, Throwable e) {
        sendWriteMessage("V", tag, msg + "\n" + getStackTrace(e));
    }

    @Override
    public void d(String tag, String msg) {
        sendWriteMessage("D", tag, msg);

    }

    @Override
    public void i(String tag, String msg) {
        sendWriteMessage("I", tag, msg);

    }

    @Override
    public void w(String tag, String msg) {
        sendWriteMessage("W", tag, msg);

    }

    @Override
    public void w(String tag, Throwable throwable) {
        sendWriteMessage("W", tag, getStackTrace(throwable));

    }

    @Override
    public void w(String tag, String msg, Throwable throwable) {
        sendWriteMessage("W", tag, msg + "\n" + getStackTrace(throwable));
    }

    @Override
    public void e(String tag, String msg) {
        sendWriteMessage("E", tag, msg);
    }

    @Override
    public void e(String tag, String msg, Throwable throwable) {
        sendWriteMessage("E", tag, msg + "\n" + getStackTrace(throwable));
    }

    @Override
    public boolean isEnable(int level) {
        return level >= mLevel;
    }
}
