package cn.ximcloud.homekit.core.proxy;


import cn.ximcloud.homekit.core.model.HomeKitAccessoryConfig;
import com.google.common.collect.Maps;
import io.github.hapjava.characteristics.HomekitCharacteristicChangeCallback;
import lombok.extern.slf4j.Slf4j;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.util.Map;

/**
 * HomeKitAccessoryInvocationHandler
 *
 * @author W9004844
 * @since 2020/01/19 14:44
 */
@Slf4j
public class HomeKitAccessoryProxy implements InvocationHandler, Serializable {

    /**
     * 引用mybatisMapperProxy对default方法和Object的方法的处理
     */
    private static final int ALLOWED_MODES = MethodHandles.Lookup.PRIVATE | MethodHandles.Lookup.PROTECTED
            | MethodHandles.Lookup.PACKAGE | MethodHandles.Lookup.PUBLIC;
    private static final Method privateLookupInMethod;
    private static final Constructor<MethodHandles.Lookup> lookupConstructor;

    static {
        Method privateLookupIn;
        try {
            privateLookupIn = MethodHandles.class.getMethod("privateLookupIn", Class.class, MethodHandles.Lookup.class);
        } catch (NoSuchMethodException e) {
            privateLookupIn = null;
        }
        privateLookupInMethod = privateLookupIn;

        Constructor<MethodHandles.Lookup> lookup = null;
        if (privateLookupInMethod == null) {
            // JDK 1.8
            try {
                lookup = MethodHandles.Lookup.class.getDeclaredConstructor(Class.class, int.class);
                lookup.setAccessible(true);
            } catch (NoSuchMethodException e) {
                throw new IllegalStateException("There is neither 'privateLookupIn(Class, Lookup)' nor 'Lookup(Class, int)' method in java.lang.invoke.MethodHandles.", e);
            } catch (Throwable t) {
                lookup = null;
            }
        }
        lookupConstructor = lookup;
    }

    /**
     * homeKitAccessory
     */
    private final HomeKitAccessoryConfig homeKitAccessoryConfig;

    /**
     * 缓存方法
     */
    private final Map<Method, HomeKitAccessoryMethod> methodCacheMap;


    /**
     * 订阅方法前缀
     */
    public static final String SUBSCRIBE_PREFIX = "subscribe";

    /**
     * 取消订阅方法前缀
     */
    public static final String UNSUBSCRIBE_PREFIX = "unsubscribe";

    /**
     * 订阅的Map
     */
    private final Map<Method, HomekitCharacteristicChangeCallback> subscribeMap = Maps.newConcurrentMap();


    /**
     * 代理对象
     *
     * @param homeKitAccessoryConfig homeKitAccessoryConfig
     * @param methodCacheMap         homeKitAccessory缓存的的方法
     */
    public HomeKitAccessoryProxy(HomeKitAccessoryConfig homeKitAccessoryConfig, Map<Method, HomeKitAccessoryMethod> methodCacheMap) {
        this.homeKitAccessoryConfig = homeKitAccessoryConfig;
        this.methodCacheMap = methodCacheMap;
    }


    /**
     * 动态代理对象执行的方法
     *
     * @param proxy  代理对象
     * @param method 执行的方法
     * @param args   执行的方法传入的参数
     * @return 返回值（真正的返回值）
     */
    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        try {
//        如果当前执行的方法的声明类指向Object则调用object的方法
            if (Object.class.equals(method.getDeclaringClass())) {
                return method.invoke(this, args);
            } else if (method.isDefault()) {
//                如果是接口default方法则执行接口default方法
                if (privateLookupInMethod == null) {
                    return invokeDefaultMethodJava8(proxy, method, args);
                } else {
                    return invokeDefaultMethodJava9(proxy, method, args);
                }
            }
        } catch (Throwable target) {
            log.error("homeKit accessory invoke method catch Error:{}", target.getMessage());
//            fix bug:一些日志类产生异常执行不到default方法，catch异常后没有捕获，执行之后的方法导致问题
            throw target;
        }
//        处理订阅和执行
        return handleSubscribeAndExecute(method, args);
    }


    private Object handleSubscribeAndExecute(Method method, Object[] args) {
        String methodName = method.getName();
        if (methodName.startsWith(SUBSCRIBE_PREFIX)) {
//            订阅
            if (subscribeMap.containsKey(method)) {
                log.warn("订阅".concat(generateMethodKey(method)).concat("已存在,调用方法:").concat(method.getDeclaringClass().getName()).concat("#").concat(methodName).concat("()"));
            } else {
                subscribeMap.put(method, (HomekitCharacteristicChangeCallback) args[0]);
                log.debug("订阅成功:{},方法:{}", generateMethodKey(method), method.getDeclaringClass().getName().concat("#").concat(methodName).concat("()"));
            }
            return null;
        } else if (methodName.startsWith(UNSUBSCRIBE_PREFIX)) {
//            取消订阅
            if (subscribeMap.containsKey(method)) {
                subscribeMap.remove(method);
                log.debug("取消订阅成功:{},调用方法:{}", generateMethodKey(method), method.getDeclaringClass().getName().concat("#").concat(methodName).concat("()"));
            } else {
                log.warn("还没有该".concat(generateMethodKey(method)).concat("订阅:").concat(method.getDeclaringClass().getName()).concat("#").concat(methodName).concat("()"));
            }
            return null;
        } else {
//            非订阅方法直接执行
            HomeKitAccessoryMethod mapperMethod = cachedMapperMethod(method);
            return mapperMethod.execute(subscribeMap, args);
        }

    }

    /**
     * 拿到缓存的方法，执行到这里就说明排除是HomeKitAccessory的父类Object或HomeKitAccessory default的方法
     * 是执行其实现接口的方法
     *
     * @param method 当前执行的方法
     * @return HomeKitAccessoryMethod 封装好的方法
     */
    private HomeKitAccessoryMethod cachedMapperMethod(Method method) {
        return methodCacheMap.computeIfAbsent(method, k -> new HomeKitAccessoryMethod(homeKitAccessoryConfig, method));
    }

    /**
     * java 9的interface 接口default方法执行
     *
     * @param proxy  代理对象
     * @param method 执行的方法
     * @param args   参数
     * @return 返回值
     * @throws Throwable 异常
     */
    private Object invokeDefaultMethodJava9(Object proxy, Method method, Object[] args)
            throws Throwable {
        final Class<?> declaringClass = method.getDeclaringClass();
        return ((MethodHandles.Lookup) privateLookupInMethod
                .invoke(null, declaringClass, MethodHandles.lookup()))
                .findSpecial(declaringClass, method.getName(), MethodType.methodType(method.getReturnType(), method.getParameterTypes()), declaringClass)
                .bindTo(proxy)
                .invokeWithArguments(args);
    }

    /**
     * java 8的interface 接口default方法执行
     *
     * @param proxy  代理对象
     * @param method 执行的方法
     * @param args   参数
     * @return 返回值
     * @throws Throwable 异常
     */
    private Object invokeDefaultMethodJava8(Object proxy, Method method, Object[] args)
            throws Throwable {
        final Class<?> declaringClass = method.getDeclaringClass();
        return lookupConstructor
                .newInstance(declaringClass, ALLOWED_MODES)
                .unreflectSpecial(method, declaringClass)
                .bindTo(proxy).invokeWithArguments(args);
    }

    private static String generateMethodKey(Method targetMethod) {
        String methodName = targetMethod.getName();
        if (methodName.startsWith(SUBSCRIBE_PREFIX)) {
            return methodName.substring(9);
        } else if (methodName.startsWith(UNSUBSCRIBE_PREFIX)) {
            return methodName.substring(11);
        }
        throw new IllegalArgumentException("方法名不正确:".concat(methodName));
    }

    @Override
    public String toString() {
        return "HomeKitAccessoryProxy{" +
                "homeKitAccessory=" + homeKitAccessoryConfig +
                ", methodCacheMap=" + methodCacheMap +
                '}';
    }
}
