package cn.ximcloud.homekit.core.proxy;

import cn.ximcloud.homekit.core.model.HomeKitAccessoryConfig;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryType;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryTypeConfig;
import io.github.hapjava.accessories.HomekitAccessory;
import io.github.hapjava.characteristics.HomekitCharacteristicChangeCallback;
import lombok.SneakyThrows;
import org.reflections.ReflectionUtils;

import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * HomeKitAccessoryMethod
 * https://www.jianshu.com/p/10e3dde8bbd6
 *
 * @author W9004844
 * @since 2020/01/20 12:48
 */
public class HomeKitAccessoryMethod {

    /**
     * homeKitAccessoryMethodCacheSet
     */
    @SuppressWarnings("unchecked")
    private static final Set<Method> homeKitAccessoryMethodCacheSet = ReflectionUtils.getMethods(HomekitAccessory.class);

    /**
     * 执行方法
     */
    private final MethodWrapper method;

    public HomeKitAccessoryMethod(HomeKitAccessoryConfig homeKitAccessoryConfig, Method method) {
//        初始化目标执行方法
        this.method = initTargetMethod(homeKitAccessoryConfig, method);
    }


    /**
     * 在此之前
     * Object里面的方法和接口default的方法都在 {@link HomeKitAccessoryProxy}
     * 的{@code invoke}方法里面被拦截执行掉了，剩下的就是真正自定义的方法
     */
    @SneakyThrows
    private MethodWrapper initTargetMethod(HomeKitAccessoryConfig homeKitAccessoryConfig, Method method) {
        if (homeKitAccessoryMethodCacheSet.contains(method)) {
//        如果是homeKitAccessory基本方法，直接调用HomeKitAccessory的方法
            return new MethodWrapper(homeKitAccessoryConfig, HomeKitAccessoryConfig.class.getDeclaredMethod(method.getName()));
        } else {
//            homeKitAccessory的实现接口声名的方法
            return new MethodWrapper(homeKitAccessoryConfig, method, findHomeKitAccessoryTypeConfig(homeKitAccessoryConfig, method));
        }
    }


    /**
     * 找到这个方法的配置
     *
     * @param homeKitAccessoryConfig accessoryConfig
     * @param method                 invoke method
     * @return HomeKitAccessoryTypeConfig
     * @throws ClassNotFoundException accessoryConfig没有这个HomekitAccessory接口的配置
     * @throws NoSuchMethodException  accessoryConfig没有找到这个方法的配置
     */
    private HomeKitAccessoryTypeConfig findHomeKitAccessoryTypeConfig(HomeKitAccessoryConfig homeKitAccessoryConfig, Method method) throws ClassNotFoundException, NoSuchMethodException {
        Class<?> declaringClass = method.getDeclaringClass();
        List<HomeKitAccessoryTypeConfig> homeKitAccessoryTypeConfigs = Optional.ofNullable(homeKitAccessoryConfig.getHomeKitAccessoryTypes().stream()
                .collect(Collectors.toMap(HomeKitAccessoryType::getTypeClass, HomeKitAccessoryType::getAccessoryTypeConfig)).get(declaringClass))
                .orElseThrow(() -> new ClassNotFoundException(declaringClass.getName()));
        return Optional.ofNullable(homeKitAccessoryTypeConfigs.stream().collect(Collectors.toMap(HomeKitAccessoryTypeConfig::getMethod, v -> v)).get(method))
                .orElseThrow(() -> new NoSuchMethodException(declaringClass.getName().concat("#").concat(method.getName()).concat("()")));
    }

    /**
     * finally execute method
     *
     * @param subscribeMap 订阅
     * @param args         请求参数
     * @return 返回值
     */
    @SneakyThrows
    public Object execute(Map<Method, HomekitCharacteristicChangeCallback> subscribeMap, Object[] args) {
        return method.invoke(subscribeMap, args);
    }


}
