package cn.ximcloud.homekit.core.proxy;

import cn.ximcloud.homekit.core.model.HomeKitAccessoryTypeConfig;
import io.github.hapjava.characteristics.HomekitCharacteristicChangeCallback;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import static cn.ximcloud.homekit.core.proxy.HomeKitAccessoryProxy.SUBSCRIBE_PREFIX;
import static cn.ximcloud.homekit.core.proxy.HomeKitAccessoryProxy.UNSUBSCRIBE_PREFIX;
import static cn.ximcloud.homekit.core.utils.CommonUtil.generateMethodString;

/**
 * MethodWrapper
 *
 * @author W9004844
 * @since 2020/01/21 14:29
 */
@Slf4j
public class MethodWrapper {

    /**
     * 目标方法
     */
    private final Method targetMethod;

    /**
     * 执行该目标方法的对象
     */
    private final Object targetObject;

    /**
     * 执行方式
     */
    private final HomeKitAccessoryTypeConfig homeKitAccessoryTypeConfig;

    /**
     * 这个方法的关键字
     */
    private final String targetMethodKey;

    /**
     * @param targetObject 执行该目标方法的对象
     * @param targetMethod 目标方法
     */
    public MethodWrapper(Object targetObject, Method targetMethod) {
        this(targetObject, targetMethod, null);
    }

    /**
     * @param targetObject               执行对象
     * @param targetMethod               执行方法
     * @param homeKitAccessoryTypeConfig 执行配置文件 如果为空则是homeKitAccessory基本方法
     */
    public MethodWrapper(Object targetObject, Method targetMethod, HomeKitAccessoryTypeConfig homeKitAccessoryTypeConfig) {
        this.targetObject = targetObject;
        this.targetMethod = targetMethod;
        this.targetMethodKey = generateMethodKey(targetMethod);
        this.homeKitAccessoryTypeConfig = homeKitAccessoryTypeConfig;
    }

    /**
     * finally execute method
     *
     * @param subscribeMap 订阅
     * @param args         执行参数
     * @return 返回值
     */
    @SneakyThrows
    public Object invoke(Map<Method, HomekitCharacteristicChangeCallback> subscribeMap, Object[] args) {
        if (homeKitAccessoryTypeConfig == null) {
            return targetMethod.invoke(targetObject, args);
        }
        Object targetReturnObject;
        switch (homeKitAccessoryTypeConfig.getInvokeType()) {
            case NOT_INVOKE:
                targetReturnObject = null;
                break;
            case VALUE:
//                TODO
                targetReturnObject = generateValueReturnObject();
                break;
            case INVOKE_METHOD:
//                TODO
                targetReturnObject = targetMethod.invoke(targetObject, args);
                break;
            case HTTP:
//                TODO
                throw new IllegalArgumentException("Call type that does not exist");
            default:
                throw new IllegalArgumentException("Call type that does not exist");
        }
//        处理订阅相关
        handleSubscription(subscribeMap);
        return targetReturnObject;
    }


    /**
     * 处理订阅相关
     */
    private void handleSubscription(Map<Method, HomekitCharacteristicChangeCallback> subscribeMap) {
//        处理订阅相关
        Map<String, HomekitCharacteristicChangeCallback> methodKeyMap = subscribeMap.entrySet().stream().collect(Collectors.toMap(k -> generateMethodKey(k.getKey()), Map.Entry::getValue));
        String targetMethodName = targetMethod.getName();
        if (methodKeyMap.containsKey(targetMethodKey) && targetMethodName.startsWith("set")) {
//            触发订阅更新
            HomekitCharacteristicChangeCallback homekitCharacteristicChangeCallback = methodKeyMap.get(targetMethodKey);
            homekitCharacteristicChangeCallback.changed();
            log.debug("invoke method:[{}], subscription[{}] update", generateMethodString(targetMethod), targetMethodKey);
        } else if (!targetMethodName.startsWith("get") && !targetMethodName.startsWith("is")) {
//            BUG:没有订阅就去获取值...
            log.debug("No such subscription:[{}],invoke method:{}", targetMethodKey, generateMethodString(targetMethod));
            log.debug("Existing subscription:{}", methodKeyMap);
        }

    }

    private Object generateValueReturnObject() throws IllegalAccessException, InstantiationException, NoSuchMethodException, InvocationTargetException {
        Class<?> returnType = targetMethod.getReturnType();
        if (CompletableFuture.class.equals(returnType)) {
            Type genericSuperclass = targetMethod.getGenericReturnType();
            if (genericSuperclass instanceof ParameterizedType) {
                Type[] actualTypeArguments = ((ParameterizedType) genericSuperclass).getActualTypeArguments();
                for (Type actualTypeArgument : actualTypeArguments) {
                    Class<?> targetReturnClass = (Class<?>) actualTypeArgument;
                    Object targetReturnObject;
                    if (!Void.class.equals(targetReturnClass)) {
                        String arg = Optional.ofNullable(homeKitAccessoryTypeConfig.getReturnArg()).orElseThrow(() -> new IllegalArgumentException("返回值不能为空")).getArg();
                        if (targetReturnClass.isEnum()) {
//                            判断是否为枚举
                            Method valueOf = targetReturnClass.getMethod("valueOf", String.class);
                            targetReturnObject = valueOf.invoke(targetReturnClass, arg);
                        } else {
                            targetReturnObject = targetReturnClass.getConstructor(String.class).newInstance(arg);
                        }
                    } else {
//                        在返回类型为void的情况,返回值永远不会用到
                        targetReturnObject = null;
                    }
                    return CompletableFuture.completedFuture(targetReturnObject);
                }
            }
            throw new UnsupportedOperationException("This return type is not supported:".concat(returnType.getTypeName()).concat(genericSuperclass.getTypeName()));
        } else if (Void.class.equals(returnType)) {
//            TODO:一些订阅操作
            return null;
        }
        throw new UnsupportedOperationException("This return type is not supported:".concat(returnType.getTypeName()));
    }


    private static String generateMethodKey(Method targetMethod) {
        String methodName = targetMethod.getName();
        if (methodName.startsWith("get")) {
            return methodName.substring(3);
        } else if (methodName.startsWith("set")) {
            return methodName.substring(3);
        } else if (methodName.startsWith(SUBSCRIBE_PREFIX)) {
            return methodName.substring(9);
        } else if (methodName.startsWith(UNSUBSCRIBE_PREFIX)) {
            return methodName.substring(11);
        }
        return methodName;
    }

}
