package cn.ximcloud.homekit.core.execution.impl;

import cn.ximcloud.homekit.core.enums.InvokeTypeEnum;
import cn.ximcloud.homekit.core.exception.GenerateReturnObjectException;
import cn.ximcloud.homekit.core.exception.UnsupportedMethodException;
import cn.ximcloud.homekit.core.execution.ExecutionScheduler;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryConfig;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryTypeConfig;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

/**
 * @author jecessewen
 */
@Slf4j
public class ValueExecutionScheduler implements ExecutionScheduler {

    @Override
    public InvokeTypeEnum getExecutionType() {
        return InvokeTypeEnum.VALUE;
    }

    @Override
    public Object execution(HomeKitAccessoryTypeConfig homeKitAccessoryTypeConfig, HomeKitAccessoryConfig targetObject, Method targetMethod, Object[] args) {
        Class<?> targetReturnClass = (Class<?>) ((ParameterizedType) targetMethod.getGenericReturnType()).getActualTypeArguments()[0];
        if (Void.class.equals(targetReturnClass)) {
//            在返回类型为void的情况,返回值永远不会用到
            return CompletableFuture.completedFuture(null);
        } else {
            Object arg = Optional.ofNullable(homeKitAccessoryTypeConfig.getReturnArg()).orElseThrow(() -> new IllegalArgumentException("The return value cannot be empty")).getArg();
            Object targetReturnObject;
            try {
                if (targetReturnClass.isEnum()) {
//                    判断是否为枚举
                    Method valueOf = targetReturnClass.getMethod("valueOf", String.class);
                    targetReturnObject = valueOf.invoke(targetReturnClass, arg.toString());
                } else {
                    targetReturnObject = targetReturnClass.getConstructor(arg.getClass()).newInstance(arg);
                }
                return CompletableFuture.completedFuture(targetReturnObject);
            } catch (NoSuchMethodException | IllegalAccessException | InstantiationException | InvocationTargetException e) {
                log.error("generate return Object error");
                throw new GenerateReturnObjectException(e);
            }
        }
    }

    @Override
    public void preCheck(HomeKitAccessoryTypeConfig homeKitAccessoryTypeConfig, HomeKitAccessoryConfig targetObject, Method targetMethod, Object[] args) {
//        check return type
        checkReturnType(targetMethod);
    }

    private void checkReturnType(Method targetMethod) {
        Class<?> returnType = targetMethod.getReturnType();
        if (!CompletableFuture.class.equals(returnType)) {
            throw new UnsupportedMethodException("This return type is not supported: ".concat(returnType.getTypeName()));
        }

        Type genericSuperclass = targetMethod.getGenericReturnType();
        if (!(genericSuperclass instanceof ParameterizedType)) {
            throw new UnsupportedOperationException("This return type is not parameterized type : ".concat(returnType.getTypeName()).concat(genericSuperclass.getTypeName()));
        }

        Type[] actualTypeArguments = ((ParameterizedType) genericSuperclass).getActualTypeArguments();
        if (actualTypeArguments.length == 0) {
            throw new UnsupportedMethodException("This return type is not supported: ".concat(returnType.getTypeName()));
        }
    }
}
