package cn.ximcloud.homekit.core.proxy;

import cn.ximcloud.homekit.core.model.HomeKitAccessoryConfig;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryType;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryTypeConfig;
import io.github.hapjava.accessories.HomekitAccessory;
import io.github.hapjava.characteristics.HomekitCharacteristicChangeCallback;
import lombok.SneakyThrows;
import org.reflections.ReflectionUtils;

import java.lang.reflect.Method;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static cn.ximcloud.homekit.core.utils.CommonUtil.generateMethodString;

/**
 * HomeKitAccessoryMethod
 * https://www.jianshu.com/p/10e3dde8bbd6
 *
 * @author W9004844
 * @since 2020/01/20 12:48
 */
public class HomeKitAccessoryMethod {

    /**
     * homeKitAccessoryMethodCacheSet
     */
    @SuppressWarnings("unchecked")
    private static final Set<Method> HOME_KIT_ACCESSORY_METHOD_CACHE_SET = ReflectionUtils.getMethods(HomekitAccessory.class);

    /**
     * 执行方法
     */
    private final MethodWrapper methodWrapper;

    public HomeKitAccessoryMethod(HomeKitAccessoryConfig homeKitAccessoryConfig, Method targetMethod) {
//        初始化目标执行方法
        this.methodWrapper = initTargetMethod(homeKitAccessoryConfig, targetMethod);
    }


    /**
     * 在此之前
     * Object里面的方法和接口default的方法都在 {@link HomeKitAccessoryProxy}
     * 的{@code invoke}方法里面被拦截执行掉了，剩下的就是真正自定义的方法
     */
    @SneakyThrows
    private MethodWrapper initTargetMethod(HomeKitAccessoryConfig homeKitAccessoryConfig, Method targetMethod) {
        if (HOME_KIT_ACCESSORY_METHOD_CACHE_SET.contains(targetMethod)) {
//        如果是homeKitAccessory基本方法，直接调用HomeKitAccessory的方法
            return new MethodWrapper(homeKitAccessoryConfig, HomeKitAccessoryConfig.class.getDeclaredMethod(targetMethod.getName(), targetMethod.getParameterTypes()));
        } else {
//            homeKitAccessory的实现接口声名的方法
            return new MethodWrapper(homeKitAccessoryConfig, targetMethod, findHomeKitAccessoryTypeConfig(homeKitAccessoryConfig, targetMethod));
        }
    }


    /**
     * 找到这个方法的配置
     *
     * @param homeKitAccessoryConfig accessoryConfig
     * @param method                 invoke method
     * @return HomeKitAccessoryTypeConfig
     * @throws ClassNotFoundException accessoryConfig没有这个HomekitAccessory接口的配置
     * @throws NoSuchMethodException  accessoryConfig没有找到这个方法的配置
     */
    private HomeKitAccessoryTypeConfig findHomeKitAccessoryTypeConfig(HomeKitAccessoryConfig homeKitAccessoryConfig, Method method) throws ClassNotFoundException, NoSuchMethodException {
        List<HomeKitAccessoryTypeConfig> homeKitAccessoryTypeConfigs = Optional.ofNullable(homeKitAccessoryConfig.getHomeKitAccessoryTypes().stream()
                .collect(Collectors.toMap(HomeKitAccessoryType::getTypeClass, HomeKitAccessoryType::getAccessoryTypeConfig)).get(method.getDeclaringClass()))
                .orElseThrow(() -> new ClassNotFoundException(method.getDeclaringClass().getName()));
        return Optional.ofNullable(homeKitAccessoryTypeConfigs.stream().collect(Collectors.toMap(HomeKitAccessoryTypeConfig::getMethod, v -> v)).get(method))
                .orElseThrow(() -> new NoSuchMethodException(String.format("not found configuration implements for method [ %s ] ,please recheck your HomeKitAccessoryTypeConfig", generateMethodString(method))));
    }

    /**
     * finally execute method 包含订阅处理
     *
     * @param homekitCharacteristicChangeCallback 订阅
     * @param args                                请求参数
     * @return 返回值
     */
    public Object invoke(HomekitCharacteristicChangeCallback homekitCharacteristicChangeCallback, Object[] args) {
        return methodWrapper.invoke(homekitCharacteristicChangeCallback, args);
    }


}
