package cn.ximcloud.homekit.core.proxy;


import cn.ximcloud.homekit.core.exception.UnsupportedMethodException;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryConfig;
import cn.ximcloud.homekit.core.utils.CommonUtil;
import com.google.common.collect.Maps;
import io.github.hapjava.characteristics.HomekitCharacteristicChangeCallback;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Map;
import java.util.Optional;

import static cn.ximcloud.homekit.core.utils.CommonUtil.generateMethodString;

/**
 * HomeKitAccessoryInvocationHandler
 *
 * @author W9004844
 * @since 2020/01/19 14:44
 */
@Slf4j
public class HomeKitAccessoryProxy implements InvocationHandler {

    /**
     * 订阅方法前缀
     */
    public static final String SUBSCRIBE_PREFIX = "subscribe";
    /**
     * 取消订阅方法前缀
     */
    public static final String UNSUBSCRIBE_PREFIX = "unsubscribe";
    /**
     * 引用mybatisMapperProxy对default方法和Object的方法的处理
     */
    private static final int ALLOWED_MODES = MethodHandles.Lookup.PRIVATE | MethodHandles.Lookup.PROTECTED
            | MethodHandles.Lookup.PACKAGE | MethodHandles.Lookup.PUBLIC;
    private static final Method PRIVATE_LOOKUP_IN_METHOD;
    private static final Constructor<MethodHandles.Lookup> LOOKUP_CONSTRUCTOR;

    static {
        Method privateLookupIn;
        try {
            privateLookupIn = MethodHandles.class.getMethod("privateLookupIn", Class.class, MethodHandles.Lookup.class);
        } catch (NoSuchMethodException e) {
            privateLookupIn = null;
        }
        PRIVATE_LOOKUP_IN_METHOD = privateLookupIn;

        Constructor<MethodHandles.Lookup> lookup = null;
        if (PRIVATE_LOOKUP_IN_METHOD == null) {
            // JDK 1.8
            try {
                lookup = MethodHandles.Lookup.class.getDeclaredConstructor(Class.class, int.class);
                lookup.setAccessible(true);
            } catch (NoSuchMethodException e) {
                throw new IllegalStateException("There is neither 'privateLookupIn(Class, Lookup)' nor 'Lookup(Class, int)' method in java.lang.invoke.MethodHandles.", e);
            } catch (Throwable t) {
                lookup = null;
            }
        }
        LOOKUP_CONSTRUCTOR = lookup;
    }

    /**
     * homeKitAccessory
     */
    private final HomeKitAccessoryConfig homeKitAccessoryConfig;
    /**
     * 缓存方法
     */
    private final Map<Method, HomeKitAccessoryMethod> methodCacheMap;
    /**
     * 每一个homekitAccessory都有一个订阅的Map
     */
    private final Map<String, HomekitCharacteristicChangeCallback> subscribeMap = Maps.newConcurrentMap();

    /**
     * 方法元数据
     */
    private final Map<Method, MethodMetaInfo> methodMetaInfoMap = Maps.newConcurrentMap();

    /**
     * 代理对象
     *
     * @param homeKitAccessoryConfig homeKitAccessoryConfig
     * @param methodCacheMap         homeKitAccessory缓存的的方法
     */
    public HomeKitAccessoryProxy(HomeKitAccessoryConfig homeKitAccessoryConfig, Map<Method, HomeKitAccessoryMethod> methodCacheMap) {
        this.homeKitAccessoryConfig = homeKitAccessoryConfig;
        this.methodCacheMap = methodCacheMap;
    }

    /**
     * 动态代理对象执行的方法
     *
     * @param proxy  代理对象
     * @param method 执行的方法
     * @param args   执行的方法传入的参数
     * @return 返回值（真正的返回值）
     */
    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        try {
//        如果当前执行的方法的声明类指向Object则调用object的方法
            if (Object.class.equals(method.getDeclaringClass())) {
                return method.invoke(this, args);
            } else if (method.isDefault()) {
//                如果是接口default方法则执行接口default方法
                if (PRIVATE_LOOKUP_IN_METHOD == null) {
                    return invokeDefaultMethodJava8(proxy, method, args);
                } else {
                    return invokeDefaultMethodJava9(proxy, method, args);
                }
            }
        } catch (Throwable target) {
            if (log.isErrorEnabled()) {
                log.error("homeKit accessory invoke method catch Error:{}", target.getMessage());
            }
//            fix bug:一些日志类产生异常执行不到default方法，catch异常后没有捕获，执行之后的方法导致问题
            throw target;
        }
//        执行并处理订阅相关
        return invokeAndHandleSubscribe(method, args);
    }

    /**
     * 执行非基本（toString、default等方法）的方法
     * 并处理订阅
     *
     * @param method 执行方法
     * @param args   请求参数
     * @return 返回值
     */
    private Object invokeAndHandleSubscribe(Method method, Object[] args) {
        MethodMetaInfo methodMetaInfo = methodMetaInfoMap.computeIfAbsent(method, MethodMetaInfo::new);
//        判断是否为订阅方法，若为订阅方法则执行订阅相关操作，不执行其他方法
        if (methodMetaInfo.isSubscribeMethod) {
            invokeSubscribeAction(methodMetaInfo, Optional.ofNullable(args).map(targetArgs -> (HomekitCharacteristicChangeCallback) targetArgs[0]).orElse(null));
            return null;
        } else {
//            非订阅方法直接执行
//            存在订阅为空的情况，即还未订阅就执行相关操作
            return cachedMapperMethod(method).invoke(Optional.ofNullable(methodMetaInfo.getMethodKey()).map(subscribeMap::get).orElse(null), args);
        }
    }

    @Data
    private static class MethodMetaInfo {

        private Method method;

        private boolean isSubscribeMethod;

        private String methodKey;

        public MethodMetaInfo(Method method) {
            this.method = method;
            this.isSubscribeMethod = isSubscribeMethod(method);
            this.methodKey = this.isSubscribeMethod ? generateMethodKey(method) : null;
        }

        public String getName() {
            return this.method.getName();
        }

        /**
         * 检查是否为订阅方法
         * 我们定义 返回值为空 方法名以 SUBSCRIBE_PREFIX 或 SUBSCRIBE_PREFIX 开头的方法为订阅方法
         *
         * @return 是否为订阅方法
         */
        private boolean isSubscribeMethod(Method method) {
            String methodName = method.getName();
            return Modifier.isPublic(method.getModifiers())
                    && ((methodName.startsWith(SUBSCRIBE_PREFIX)
                    && method.getParameters().length == 1
                    && method.getParameters()[0].getType().equals(HomekitCharacteristicChangeCallback.class))
                    || methodName.startsWith(UNSUBSCRIBE_PREFIX))
                    && method.getReturnType().equals(Void.TYPE);
        }

        private static String generateMethodKey(Method targetMethod) {
            String methodName = targetMethod.getName();
            if (methodName.startsWith(SUBSCRIBE_PREFIX)) {
                return methodName.substring(9);
            } else if (methodName.startsWith(UNSUBSCRIBE_PREFIX)) {
                return methodName.substring(11);
            }
            throw new IllegalArgumentException("Incorrect method name:".concat(methodName));
        }

    }

    private void invokeSubscribeAction(MethodMetaInfo methodMetaInfo, HomekitCharacteristicChangeCallback homekitCharacteristicChangeCallback) {
        Method method = methodMetaInfo.getMethod();
        String methodName = methodMetaInfo.getName();
        String methodKey = methodMetaInfo.getMethodKey();
        if (methodName.startsWith(SUBSCRIBE_PREFIX) && homekitCharacteristicChangeCallback != null) {
//            订阅
            if (subscribeMap.containsKey(methodKey)) {
                if (log.isWarnEnabled()) {
                    log.warn("subscription [{}] is existed,invoke method:{}", methodKey, generateMethodString(method));
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Successfully subscribed:{},method:{}", methodKey, generateMethodString(method));
                }
            }
            subscribeMap.put(methodKey, homekitCharacteristicChangeCallback);
        } else if (methodName.startsWith(UNSUBSCRIBE_PREFIX) && homekitCharacteristicChangeCallback == null) {
//            取消订阅
            if (subscribeMap.containsKey(methodKey)) {
                subscribeMap.remove(methodKey);
                if (log.isDebugEnabled()) {
                    log.debug("Unsubscribe successfully:{},invoke method:{}", methodKey, generateMethodString(method));
                }
            } else {
                if (log.isWarnEnabled()) {
                    log.warn("[{}] Not yet subscribed,method:{}", methodKey, generateMethodString(method));
                }
            }
        } else {
            throw new UnsupportedMethodException(CommonUtil.generateMethodString(method));
        }
    }

    /**
     * 拿到缓存的方法，执行到这里就说明排除是HomeKitAccessory的父类Object或HomeKitAccessory default的方法
     * 是执行其实现接口的方法
     *
     * @param method 当前执行的方法
     * @return HomeKitAccessoryMethod 封装好的方法
     */
    private HomeKitAccessoryMethod cachedMapperMethod(Method method) {
        return methodCacheMap.computeIfAbsent(method, targetMethod -> new HomeKitAccessoryMethod(homeKitAccessoryConfig, targetMethod));
    }

    /**
     * java 9的interface 接口default方法执行
     *
     * @param proxy  代理对象
     * @param method 执行的方法
     * @param args   参数
     * @return 返回值
     * @throws Throwable 异常
     */
    private Object invokeDefaultMethodJava9(Object proxy, Method method, Object[] args)
            throws Throwable {
        final Class<?> declaringClass = method.getDeclaringClass();
        return ((MethodHandles.Lookup) PRIVATE_LOOKUP_IN_METHOD
                .invoke(null, declaringClass, MethodHandles.lookup()))
                .findSpecial(declaringClass, method.getName(), MethodType.methodType(method.getReturnType(), method.getParameterTypes()), declaringClass)
                .bindTo(proxy)
                .invokeWithArguments(args);
    }

    /**
     * java 8的interface 接口default方法执行
     *
     * @param proxy  代理对象
     * @param method 执行的方法
     * @param args   参数
     * @return 返回值
     * @throws Throwable 异常
     */
    private Object invokeDefaultMethodJava8(Object proxy, Method method, Object[] args)
            throws Throwable {
        final Class<?> declaringClass = method.getDeclaringClass();
        return LOOKUP_CONSTRUCTOR
                .newInstance(declaringClass, ALLOWED_MODES)
                .unreflectSpecial(method, declaringClass)
                .bindTo(proxy).invokeWithArguments(args);
    }

    @Override
    public String toString() {
        return "HomeKitAccessoryProxy{" +
                "homeKitAccessory=" + homeKitAccessoryConfig +
                '}';
    }
}
