package cn.ximcloud.homekit.core.proxy;

import cn.ximcloud.homekit.core.model.HomeKitAccessoryConfig;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryType;
import com.google.common.collect.Maps;
import io.github.hapjava.accessories.HomekitAccessory;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Map;
import java.util.Optional;

/**
 * HomeKitProxyFactory
 *
 * @author W9004844
 * @since 2020/01/19 14:26
 */
public class HomeKitAccessoryProxyFactory {

    private static final String ACCESS_PACKAGE_PREFIX = "io.github.hapjava.accessories";

    /**
     * HomeKitAccessory数据库对象
     */
    private final HomeKitAccessoryConfig homeKitAccessoryConfig;

    /**
     * 不止有一个接口
     */
    private final Class<?>[] homeKitAccessoryInterfaces;

    /**
     * 执行方法的Map
     */
    private final Map<Method, HomeKitAccessoryMethod> homeKitAccessoryMethodMap = Maps.newConcurrentMap();

    public HomeKitAccessoryProxyFactory(HomeKitAccessoryConfig homeKitAccessoryConfig) {
        this.homeKitAccessoryInterfaces = Optional.ofNullable(homeKitAccessoryConfig.getHomeKitAccessoryTypes())
                .orElseThrow(() -> new IllegalArgumentException("homekit accessory type list can not be empty!"))
                .stream()
                .map(HomeKitAccessoryType::getTypeClass)
                .peek(HomeKitAccessoryProxyFactory::checkClass)
                .toArray(Class[]::new);
        this.homeKitAccessoryConfig = homeKitAccessoryConfig;
    }

    private static void checkClass(Class<?> clazz) {
        String className = clazz.getName();
        if (!className.startsWith(ACCESS_PACKAGE_PREFIX)) {
            throw new IllegalArgumentException("this class ".concat(className).concat("is not supported"));
        }
    }

    public Class<?>[] getHomeKitAccessoryInterfaces() {
        return homeKitAccessoryInterfaces;
    }

    public Map<Method, HomeKitAccessoryMethod> getHomeKitAccessoryMethodMap() {
        return homeKitAccessoryMethodMap;
    }

    protected HomekitAccessory newInstance(HomeKitAccessoryProxy homeKitAccessoryProxy) {
        return (HomekitAccessory) Proxy.newProxyInstance(HomekitAccessory.class.getClassLoader(), homeKitAccessoryInterfaces, homeKitAccessoryProxy);
    }

    public HomekitAccessory newInstance() {
        return newInstance(new HomeKitAccessoryProxy(homeKitAccessoryConfig, homeKitAccessoryMethodMap));
    }

}
