package cn.ximcloud.homekit.core.proxy;

import cn.ximcloud.homekit.core.exception.UnsupportedMethodException;
import cn.ximcloud.homekit.core.execution.ExecutionSchedulerBuilder;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryConfig;
import cn.ximcloud.homekit.core.model.HomeKitAccessoryTypeConfig;
import cn.ximcloud.homekit.core.utils.CommonUtil;
import io.github.hapjava.characteristics.HomekitCharacteristicChangeCallback;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import static cn.ximcloud.homekit.core.utils.CommonUtil.generateMethodString;

/**
 * MethodWrapper
 *
 * @author W9004844
 * @since 2020/01/21 14:29
 */
@Slf4j
public class MethodWrapper {

    private static final String PREFIX_GET = "get";
    private static final String PREFIX_SET = "set";
    private static final String PREFIX_IS = "is";
    /**
     * 目标方法
     */
    private final Method targetMethod;
    /**
     * 执行该目标方法的对象
     */
    private final HomeKitAccessoryConfig targetObject;
    /**
     * 执行方式
     */
    private final HomeKitAccessoryTypeConfig homeKitAccessoryTypeConfig;

    /**
     * @param targetObject 执行该目标方法的对象
     * @param targetMethod 目标方法
     */
    public MethodWrapper(HomeKitAccessoryConfig targetObject, Method targetMethod) {
        this(targetObject, targetMethod, null);
    }

    /**
     * @param targetObject               执行对象
     * @param targetMethod               执行方法
     * @param homeKitAccessoryTypeConfig 执行配置文件 如果为空则是homeKitAccessory基本方法
     */
    public MethodWrapper(final HomeKitAccessoryConfig targetObject, final Method targetMethod, final HomeKitAccessoryTypeConfig homeKitAccessoryTypeConfig) {
        this.targetObject = targetObject;
        this.targetMethod = targetMethod;
        this.homeKitAccessoryTypeConfig = homeKitAccessoryTypeConfig;
    }

    /**
     * finally execute method
     *
     * @param homekitCharacteristicChangeCallback 订阅
     * @param args                                执行参数
     * @return 返回值
     */
    public Object invoke(HomekitCharacteristicChangeCallback homekitCharacteristicChangeCallback, Object[] args) {
        return Optional.ofNullable(homeKitAccessoryTypeConfig)
                .map(config -> invokeImplementMethodAndHandleSubscriptionAction(homekitCharacteristicChangeCallback, args))
                .orElseGet(() -> invokeHomeKitAccessoryBasicMethod(args));
    }

    /**
     * homeKitAccessory的实现接口声名的方法
     *
     * @param args 请求参数
     * @return 返回值
     */
    private Object invokeImplementMethodAndHandleSubscriptionAction(HomekitCharacteristicChangeCallback homekitCharacteristicChangeCallback, Object[] args) {
//            homeKitAccessory的实现接口声名的方法
        Object targetReturnObject = ExecutionSchedulerBuilder
                .getExecutionSchedule(homeKitAccessoryTypeConfig.getInvokeType())
                .execution0(homeKitAccessoryTypeConfig, targetObject, targetMethod, args);
//        异步处理订阅相关
        CompletableFuture.runAsync(() -> handleSubscriptionAction(homekitCharacteristicChangeCallback));
        return targetReturnObject;
    }

    /**
     * 执行homeKitAccessory基本方法
     * 若homeKitAccessoryTypeConfig 为空，则是基本方法
     *
     * @param args 请求参数
     * @return 返回值
     */
    private Object invokeHomeKitAccessoryBasicMethod(Object[] args) {
        try {
            return targetMethod.invoke(targetObject, args);
        } catch (IllegalAccessException | InvocationTargetException e) {
            throw new UnsupportedMethodException("unsupported method: ".concat(CommonUtil.generateMethodString(targetMethod)), e);
        }
    }

    /**
     * 处理订阅相关
     */
    private void handleSubscriptionAction(HomekitCharacteristicChangeCallback homekitCharacteristicChangeCallback) {
//        处理订阅相关
        if (homekitCharacteristicChangeCallback != null) {
//            触发订阅更新
            homekitCharacteristicChangeCallback.changed();
            if (log.isDebugEnabled()) {
                log.debug("invoke method:[{}], callback changed", generateMethodString(targetMethod));
            }
        } else {
//            BUG:没有订阅就去获取值...
            if (log.isDebugEnabled()) {
                log.debug("No such subscription,invoke method:{}", generateMethodString(targetMethod));
            }
        }
    }

}
