package cn.ximcloud.homekit.core.starter.auth;

import cn.ximcloud.homekit.core.starter.entity.HomeKitAuthInfoEntity;
import cn.ximcloud.homekit.core.starter.entity.HomeKitUserEntity;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAuthInfoRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitUserRepository;
import io.github.hapjava.server.HomekitAuthInfo;
import io.github.hapjava.server.impl.HomekitServer;
import io.github.hapjava.server.impl.HomekitUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.util.Optional;


/**
 * XIMCloudHomeKitAuthInfo
 *
 * @author W9004844
 */
@Slf4j
@AllArgsConstructor
public class BaseOnDataBaseHomeKitAuthInfo implements HomekitAuthInfo {

    private final HomeKitUserRepository homeKitUserRepository;

    private final HomeKitAuthInfoEntity homeKitAuthInfoEntity;

    public BaseOnDataBaseHomeKitAuthInfo(HomeKitAuthInfoRepository homeKitAuthInfoRepository, HomeKitUserRepository homeKitUserRepository) {
        this.homeKitUserRepository = homeKitUserRepository;

        Optional<HomeKitAuthInfoEntity> homeKitAuthInfoEntity = homeKitAuthInfoRepository.findById(1);
        HomeKitAuthInfoEntity homeKitAuthInfo = homeKitAuthInfoEntity.orElseGet(() -> {
            try {
                return new HomeKitAuthInfoEntity(
                        HomekitServer.generatePin(),
                        HomekitServer.generateMac(),
                        HomekitServer.generateSalt(),
                        HomekitUtils.generateKey());
            } catch (InvalidAlgorithmParameterException e) {
                throw new RuntimeException("generate HomeKitAuthInfo error", e);
            }
        });
        homeKitAuthInfoRepository.update(homeKitAuthInfo);
        this.homeKitAuthInfoEntity = homeKitAuthInfo;
        log.info("The PIN for pairing is {}", homeKitAuthInfo.getPin());
    }

    @Override
    public String getPin() {
        return homeKitAuthInfoEntity.getPin();
    }

    @Override
    public String getMac() {
        return homeKitAuthInfoEntity.getMac();
    }

    @Override
    public BigInteger getSalt() {
        return homeKitAuthInfoEntity.getSalt();
    }

    @Override
    public byte[] getPrivateKey() {
        return homeKitAuthInfoEntity.getPrivateKey();
    }

    @Override
    public void createUser(String username, byte[] publicKey) {
        Optional<HomeKitUserEntity> homeKitUser = homeKitUserRepository.findByUserName(username);
        if (!homeKitUser.isPresent()) {
            homeKitUserRepository.save(new HomeKitUserEntity(username, publicKey));
            log.info("Added user for {}", username);
        } else {
            log.warn("Already have an user for {}", username);
        }
    }

    @Override
    public void removeUser(String username) {
        Optional<HomeKitUserEntity> homeKitUser = homeKitUserRepository.findByUserName(username);
        if (homeKitUser.isPresent()) {
            log.info("Removed user for {}", username);
            homeKitUserRepository.delete(homeKitUser.get());
        } else {
            log.warn("the user {} not found", username);
        }
    }

    @Override
    public byte[] getUserPublicKey(String username) {
        Optional<HomeKitUserEntity> homeKitUser = homeKitUserRepository.findByUserName(username);
        if (homeKitUser.isPresent()) {
            return homeKitUser.get().getPublicKey();
        } else {
            log.warn("the user {} not found", username);
            return null;
        }
    }

    /**
     * Called to check if a user has been created. The homekit accessory advertises whether the
     * accessory has already been paired. At this time, it's unclear whether multiple users can be
     * created, however it is known that advertising as unpaired will break in iOS 9. The default
     * value has been provided to maintain API compatibility for implementations targeting iOS 8.
     *
     * @return whether a user has been created and stored
     */
    @Override
    public boolean hasUser() {
        return homeKitUserRepository.count() > 0;
    }
}
