package cn.ximcloud.homekit.core.starter.auth;

import io.github.hapjava.server.HomekitAuthInfo;
import io.github.hapjava.server.impl.HomekitServer;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.io.*;
import java.math.BigInteger;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * XIMCloudHomeKitAuthInfo
 *
 * @author W9004844
 */
@Slf4j
public class SimpleDemoHomeKitAuthInfo implements HomekitAuthInfo {

    /**
     * pin
     */
    private final String pin;

    /**
     * mac
     */
    private final String mac;

    /**
     * salt
     */
    private final BigInteger salt;

    /**
     * privateKey
     */
    private final byte[] privateKey;

    /**
     * userKeyMap
     */
    private Map<String, byte[]> userMap = new ConcurrentHashMap<>();

    private final transient File authFile = new File("auth-state.bin");

    @SneakyThrows
    public SimpleDemoHomeKitAuthInfo() {
        if (authFile.exists()) {
            try (ObjectInputStream objectInputStream = new ObjectInputStream(new FileInputStream(authFile))) {
                System.out.println("Using persisted auth");
                SimpleDemoHomeKitAuthInfo simpleDemoHomeKitAuthInfo = (SimpleDemoHomeKitAuthInfo) objectInputStream.readObject();
                this.pin = simpleDemoHomeKitAuthInfo.pin;
                this.mac = simpleDemoHomeKitAuthInfo.mac;
                this.salt = simpleDemoHomeKitAuthInfo.salt;
                this.privateKey = simpleDemoHomeKitAuthInfo.privateKey;
                this.userMap = simpleDemoHomeKitAuthInfo.userMap;
            }
        } else {
            pin = HomekitServer.generatePin();
            mac = HomekitServer.generateMac();
            salt = HomekitServer.generateSalt();
            privateKey = HomekitServer.generateKey();
            notifyChanged();
        }
        log.info("The PIN for pairing is {}", pin);
    }

    private void notifyChanged() {
        try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(new FileOutputStream(authFile))) {
            objectOutputStream.writeObject(this);
            objectOutputStream.flush();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * A pin code used for pairing the device. This pin will be required within iOS in order to
     * complete pairing. The numbers cannot be sequential and should not have a repeating pattern.
     *
     * @return the pin code, in the form ###-##-###
     */
    @Override
    public String getPin() {
        return pin;
    }

    /**
     * A unique MAC address to be advertised with the HomeKit information. This does not have to be
     * the MAC address of the network interface. You can generate this using {@link
     * HomekitServer#generateMac()}.
     *
     * @return the unique MAC address.
     */
    @Override
    public String getMac() {
        return mac;
    }

    /**
     * The Salt that will be used when hashing the pin code to send to the client. You should generate
     * this using {@link HomekitServer#generateSalt()}.
     *
     * @return the Salt.
     */
    @Override
    public BigInteger getSalt() {
        return salt;
    }

    /**
     * The private key used by the server during pairing and message encryption. You should generate
     * this using {@link HomekitServer#generateKey()}
     *
     * @return the private key.
     */
    @Override
    public byte[] getPrivateKey() {
        return privateKey;
    }

    @Override
    public void createUser(String username, byte[] publicKey) {
        if (!userMap.containsKey(username)) {
            userMap.putIfAbsent(username, publicKey);
            log.info("Added user for {}", username);
        } else {
            log.warn("Already have this user for {}", username);
        }
        notifyChanged();
    }

    @Override
    public void removeUser(String username) {
        if (userMap.containsKey(username)) {
            userMap.remove(username);
            log.info("Removed pairing for {}", username);
        } else {
            log.warn("the user {} not found", username);
        }
        notifyChanged();
    }

    @Override
    public byte[] getUserPublicKey(String username) {
        if (userMap.containsKey(username)) {
            return userMap.get(username);
        } else {
            log.warn("the user {} not found", username);
            return null;
        }
    }

    /**
     * Called to check if a user has been created. The homekit accessory advertises whether the
     * accessory has already been paired. At this time, it's unclear whether multiple users can be
     * created, however it is known that advertising as unpaired will break in iOS 9. The default
     * value has been provided to maintain API compatibility for implementations targeting iOS 8.
     *
     * @return whether a user has been created and stored
     */
    @Override
    public boolean hasUser() {
        return !userMap.isEmpty();
    }

}
