package cn.ximcloud.homekit.core.starter.core.service.impl;

import cn.ximcloud.homekit.core.starter.auth.BaseOnDataBaseHomeKitAuthInfo;
import cn.ximcloud.homekit.core.starter.config.HomeKitConfig;
import cn.ximcloud.homekit.core.starter.config.HomeKitDataSourceConfig;
import cn.ximcloud.homekit.core.starter.constants.HomeKitAccessoryTypeEnum;
import cn.ximcloud.homekit.core.starter.core.accessories.DemoLightAccessory;
import cn.ximcloud.homekit.core.starter.core.accessories.DemoSwitchAccessory;
import cn.ximcloud.homekit.core.starter.core.service.HomeKitService;
import cn.ximcloud.homekit.core.starter.entity.HomeKitAccessoryConfigEntity;
import cn.ximcloud.homekit.core.starter.entity.metadata.HomeKitAccessoryEntity;
import cn.ximcloud.homekit.core.starter.entity.metadata.HomeKitAccessoryMethodEntity;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAccessoryConfigRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAccessoryRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAuthInfoRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitUserRepository;
import cn.ximcloud.homekit.core.starter.utils.HomeKitAccessoryScanner;
import cn.ximcloud.homekit.core.starter.utils.HomekitAccessoryBuilder;
import cn.ximcloud.homekit.core.starter.utils.HomekitAccessoryCoverUtil;
import io.github.hapjava.accessories.HomekitAccessory;
import io.github.hapjava.server.HomekitAuthInfo;
import lombok.extern.slf4j.Slf4j;
import org.reflections.ReflectionUtils;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import static cn.ximcloud.homekit.core.starter.utils.HomekitAccessoryCoverUtil.coverToHomeKitAccessoryMethodEntity;


@Slf4j
public class BaseOnDataBaseHomeKitService implements HomeKitService {

    private final HomeKitAccessoryConfigRepository homeKitAccessoryConfigRepository;

    private final HomeKitConfig homeKitConfig;

    private final HomeKitAuthInfoRepository homeKitAuthInfoRepository;

    private final HomeKitUserRepository homeKitUserRepository;

    private final HomeKitDataSourceConfig homeKitDataSourceConfig;

    private final HomeKitAccessoryRepository homeKitAccessoryRepository;

    private final HomekitAccessoryBuilder homekitAccessoryBuilder;

    public BaseOnDataBaseHomeKitService(HomeKitAuthInfoRepository homeKitAuthInfoRepository,
                                        HomeKitUserRepository homeKitUserRepository,
                                        HomeKitConfig homeKitConfig,
                                        HomeKitAccessoryConfigRepository homeKitAccessoryConfigRepository,
                                        HomeKitDataSourceConfig homeKitDataSourceConfig,
                                        HomeKitAccessoryRepository homeKitAccessoryRepository,
                                        HomekitAccessoryBuilder homekitAccessoryBuilder) {
        this.homeKitAuthInfoRepository = homeKitAuthInfoRepository;
        this.homeKitUserRepository = homeKitUserRepository;
        this.homeKitConfig = homeKitConfig;
        this.homeKitAccessoryConfigRepository = homeKitAccessoryConfigRepository;
        this.homeKitDataSourceConfig = homeKitDataSourceConfig;
        this.homeKitAccessoryRepository = homeKitAccessoryRepository;
        this.homekitAccessoryBuilder = homekitAccessoryBuilder;
        handlerHomeKitAccessoryMetaData();
    }

    /**
     * 获取HomeKit 配置
     *
     * @return MockAuthInfo
     */
    @Override
    public HomekitAuthInfo getAuthInfo() {
        return new BaseOnDataBaseHomeKitAuthInfo(homeKitAuthInfoRepository, homeKitUserRepository);
    }


    /**
     * 设备列表
     *
     * @return List<HomeKitAccessory>
     */
    @Override
    public List<HomekitAccessory> getAccessories() {
        List<HomekitAccessory> accessories = new ArrayList<>();
        addDefaultAccessories(accessories);
        return accessories;
    }

    @Transactional(rollbackFor = Exception.class)
    public void handlerHomeKitAccessoryMetaData() {
        HomeKitAccessoryScanner.scanHomeKitAccessory()
                .forEach(homeKitAccessoryClass -> extracted(homeKitAccessoryClass, HomeKitAccessoryTypeEnum.ACCESSORY));

        HomeKitAccessoryScanner.scanHomeKitAccessoryOptionalCharacteristic()
                .forEach(optionalCharacteristic -> extracted(optionalCharacteristic, HomeKitAccessoryTypeEnum.OPTIONAL_CHARACTERISTIC));
    }

    private void extracted(Class<?> homeKitAccessoryClass, HomeKitAccessoryTypeEnum type) {
        if (!homeKitAccessoryRepository.findByClazz(homeKitAccessoryClass).isPresent()) {
            HomeKitAccessoryEntity entity = new HomeKitAccessoryEntity();
            entity.setClazz(homeKitAccessoryClass);
            entity.setType(type);
            List<HomeKitAccessoryMethodEntity> methodList = new ArrayList<>();
            @SuppressWarnings("unchecked")
            Set<Method> allMethods = ReflectionUtils.getAllMethods(homeKitAccessoryClass);
            allMethods.forEach(method -> {
                HomeKitAccessoryMethodEntity homeKitAccessoryMethodEntity = coverToHomeKitAccessoryMethodEntity(method);
                homeKitAccessoryMethodEntity.setHomeKitAccessory(entity);
                methodList.add(homeKitAccessoryMethodEntity);
            });
            entity.setHomekitAccessoryMethodList(methodList);
            homeKitAccessoryRepository.save(entity);
        }
    }

    /**
     * 添加默认HomeKitAccessory配件
     *
     * @param accessories
     */
    private void addDefaultAccessories(List<HomekitAccessory> accessories) {
        generateDemoAccessoriesWithCondition(accessories);
        generateFromDataBaseAccessories(accessories);
    }

    /**
     * 添加demo配件
     *
     * @param accessories
     */
    private void generateDemoAccessoriesWithCondition(List<HomekitAccessory> accessories) {
        if (homeKitConfig.isAddDemo()) {
            accessories.add(DemoSwitchAccessory.generateSimpleDemoSwitchAccessory());
            accessories.add(new DemoLightAccessory());
        }
    }

    /**
     * 添加数据库中的配件
     *
     * @param accessories
     */
    @Transactional(rollbackFor = Exception.class)
    public void generateFromDataBaseAccessories(List<HomekitAccessory> accessories) {
        if (homeKitConfig.isAddDemo()) {
            if (homeKitAccessoryConfigRepository.count() == 0) {
                homeKitAccessoryConfigRepository.save(homekitAccessoryBuilder.buildDemoHomeKitAccessory());
            }
        }
        Iterable<HomeKitAccessoryConfigEntity> homeKitAccessoryConfigEntities = homeKitAccessoryConfigRepository.findAll();
        homeKitAccessoryConfigEntities.forEach(homeKitAccessoryConfigEntity ->
                accessories.add(HomekitAccessoryCoverUtil.coverToHomekitAccessory(homeKitAccessoryConfigEntity)));

    }


}
