package cn.ximcloud.homekit.core.starter.autoconfig;

import cn.ximcloud.homekit.core.starter.constants.DataSourceConstant;
import cn.ximcloud.homekit.core.starter.autoconfig.condition.EmbeddedDataBaseCondition;
import cn.ximcloud.homekit.core.starter.autoconfig.condition.SimpleDemoHomeKitCondition;
import cn.ximcloud.homekit.core.starter.autoconfig.properties.HomeKitDataSourceProperties;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.domain.EntityScan;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.jpa.repository.config.EnableJpaRepositories;
import org.springframework.stereotype.Component;

import javax.sql.DataSource;
import java.util.Optional;

@Slf4j
@Component
@Configuration
@AutoConfigureAfter(HomeKitDataSourceProperties.class)
@EntityScan("cn.ximcloud.homekit.core.starter.entity")
@EnableJpaRepositories("cn.ximcloud.homekit.core.starter.repository")
@AutoConfigureBefore(value = {DataSource.class, HomeKitServiceAutoConfiguration.class})
public class HomeKitDataAutoConfiguration {

    static {
        log.info("init datasource auto configuration");
    }

    private final HomeKitDataSourceProperties homeKitDataSourceProperties;

    public HomeKitDataAutoConfiguration(HomeKitDataSourceProperties homeKitDataSourceProperties) {
        this.homeKitDataSourceProperties = homeKitDataSourceProperties;
    }


    /**
     * 当DataSource缺失，并且为SimpleDemo模式。generate一个不会使用的DataSource....
     * <p>
     * BUG:
     * ximcloud-homekit-starter-demo可以看到...由于autoconfig没有数据源都要给弄一个数据源...
     * 想要实现的效果是，如果确实DataSource，并且也为简单模式，怎么实现不自动注入Jpa相关的东西
     *
     * @return DataSource
     */
    @Bean
    @ConditionalOnMissingBean(DataSource.class)
    @Conditional(SimpleDemoHomeKitCondition.class)
    public DataSource generateUnUsedDataSource() {
        log.info("generate unused DataSource");
        HikariConfig hikariConfig = new HikariConfig();
        hikariConfig.setDriverClassName("org.h2.Driver");
        hikariConfig.setJdbcUrl("jdbc:h2:mem:unused");
        hikariConfig.setUsername("sa");
        hikariConfig.setPassword("sa");
        return new HikariDataSource(hikariConfig);
    }

    @Bean
    @ConditionalOnMissingBean(DataSource.class)
    @Conditional(EmbeddedDataBaseCondition.class)
    public DataSource generateEmbeddedDataSource() {
        log.info("generate Embedded DataSource");
        HikariConfig hikariConfig = new HikariConfig();
        hikariConfig.setDriverClassName("org.h2.Driver");
        String jdbcUrl = generateJdbcUrl();
        hikariConfig.setJdbcUrl(jdbcUrl);
        hikariConfig.setUsername("sa");
        hikariConfig.setPassword("sa");
        log.info("the jdbc url is:[{}] ,username:[{}],password:[{}]", jdbcUrl, hikariConfig.getUsername(), hikariConfig.getPassword());
        return new HikariDataSource(hikariConfig);
    }

    private String generateJdbcUrl() {
        if (DataSourceConstant.EMBEDDED.equals(homeKitDataSourceProperties.getType())) {
            return "jdbc:h2:mem:ximcloud_homekit";
        } else {
            return "jdbc:h2:file:".concat(Optional.ofNullable(homeKitDataSourceProperties.getFile()).orElse("./db/ximcloud_homekit"));
        }
    }

}
