package cn.ximcloud.homekit.core.starter.core.accessories;

import cn.ximcloud.homekit.core.starter.constants.PlatformEnum;
import cn.ximcloud.homekit.core.starter.util.PingUtil;
import cn.ximcloud.homekit.core.starter.util.ShutdownUtil;
import io.github.hapjava.accessories.SwitchAccessory;
import io.github.hapjava.accessories.optionalcharacteristic.AccessoryWithHardwareRevision;
import io.github.hapjava.characteristics.HomekitCharacteristicChangeCallback;
import lombok.extern.slf4j.Slf4j;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

@Slf4j
public class WolSwitch implements SwitchAccessory, AccessoryWithHardwareRevision {

    /**
     * machine ipaddress also can be domain
     */
    private String ipAddress;

    /**
     * machine address
     */
    private String macAddress;

    private final int id;

    private final PlatformEnum platform;

    private long time = 30;

    private TimeUnit timeUnit = TimeUnit.SECONDS;

    private HomekitCharacteristicChangeCallback subscribeCallback = null;


    @Override
    public int getId() {
        return id;
    }

    public WolSwitch(int id, String ipAddress, String macAddress) {
        this(id, ipAddress, macAddress, PlatformEnum.WINDOWS);
    }

    public WolSwitch(int id, String ipAddress, String macAddress, PlatformEnum platform) {

        this.id = id;
        this.ipAddress = ipAddress;
        this.macAddress = macAddress;
        this.platform = platform;
    }

    public String getIpAddress() {
        return ipAddress;
    }

    public void setIpAddress(String ipAddress) {
        this.ipAddress = ipAddress;
    }

    public String getMacAddress() {
        return macAddress;
    }

    public void setMacAddress(String macAddress) {
        this.macAddress = macAddress;
    }

    @Override
    public CompletableFuture<String> getName() {
        return CompletableFuture.completedFuture("Computer WOL Switch");
    }

    @Override
    public void identify() {
        System.out.println("this is a Computer WOL switch accessory");
    }

    @Override
    public CompletableFuture<String> getSerialNumber() {
        return CompletableFuture.completedFuture("Test SwitchSN");
    }

    @Override
    public CompletableFuture<String> getModel() {
        return CompletableFuture.completedFuture("TestSwitch Model");
    }

    @Override
    public CompletableFuture<String> getManufacturer() {
        return CompletableFuture.completedFuture("Test SwitchManufacturer");
    }

    @Override
    public CompletableFuture<String> getFirmwareRevision() {
        return CompletableFuture.completedFuture("Test Switch Firmware");
    }

    @Override
    public CompletableFuture<String> getHardwareRevision() {
        return CompletableFuture.completedFuture("Test Switch Hardware");
    }

    /**
     * Retrieves the current binary state of the switch.
     *
     * @return a future that will contain the binary state
     */
    @Override
    public CompletableFuture<Boolean> getSwitchState() {
        return PingUtil.ping(ipAddress);
    }

    /**
     * Sets the binary state of the switch
     *
     * @param state the binary state to set
     * @return a future that completes when the change is made
     * @throws Exception when the change cannot be made
     */
    @Override
    public CompletableFuture<Void> setSwitchState(boolean state) throws Exception {
        if (getSwitchState().join()) {
//            开机状态
            if (state) {
                log.debug("the computer is power on now,not need to wol");
            } else {
                ShutdownUtil.shutdown(ipAddress, platform, timeUnit.toSeconds(time));
                Optional.ofNullable(subscribeCallback).ifPresent(HomekitCharacteristicChangeCallback::changed);
            }
        } else {
//            关机状态
            if (state) {
                //        do something
                Optional.ofNullable(subscribeCallback).ifPresent(HomekitCharacteristicChangeCallback::changed);
            } else {
                log.debug("the computer is power off now,not need to power off");
            }

        }
        return CompletableFuture.completedFuture(null);
    }

    /**
     * Subscribes to changes in the binary state of the switch.
     *
     * @param callback the function to call when the state changes.
     */
    @Override
    public void subscribeSwitchState(HomekitCharacteristicChangeCallback callback) {
        this.subscribeCallback = callback;
    }

    /**
     * Unsubscribes from changes in the binary state of the switch.
     */
    @Override
    public void unsubscribeSwitchState() {
        this.subscribeCallback = null;
    }
}
