package cn.ximcloud.homekit.core.starter.core.accessories;

import cn.ximcloud.homekit.core.starter.constants.PlatformEnum;
import cn.ximcloud.homekit.core.starter.util.wol.PingUtil;
import cn.ximcloud.homekit.core.starter.util.wol.ShutdownAndWakeUpUtil;
import cn.ximcloud.homekit.core.starter.util.wol.WOLNode;
import cn.ximcloud.homekit.core.starter.util.wol.validator.MacAddressValidator;
import io.github.hapjava.accessories.SwitchAccessory;
import io.github.hapjava.accessories.optionalcharacteristic.AccessoryWithHardwareRevision;
import io.github.hapjava.characteristics.HomekitCharacteristicChangeCallback;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.io.Serializable;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

/**
 * @author jecessewen
 */
@Slf4j
public class WolSwitch implements SwitchAccessory, AccessoryWithHardwareRevision, Serializable {

    /**
     * machine ipaddress also can be domain
     * 用于去ping看设备是否在线
     */
    private String ipAddress;

    /**
     * machine address
     * 用于远程唤醒设备
     * MAC Address allowed formats
     *
     * 11:22:33:AA:BB:CC
     * 11-22-33-AA-BB-CC
     * 1122.33AA.BBCC
     * 112233AABBCC
     */
    private String macAddress;

    /**
     * HomeKit Accessory Id
     */
    private final int id;

    /**
     * 平台类型
     */
    private final PlatformEnum platform;

    /**
     * 超时时间
     */
    private long timeOut;

    /**
     * 超时时间
     */
    private TimeUnit timeUnit;

    /**
     * 回调
     */
    private HomekitCharacteristicChangeCallback subscribeCallback;

    /**
     * 重试
     */
    private int retry;

    private String broadcastIp;

    private int broadcastPort;


    @Override
    public int getId() {
        return id;
    }

    /**
     * 默认平台Windows
     *
     * @param id         HomeKit Accessory Id
     * @param ipAddress  ipAddress
     * @param macAddress macAddress
     */
    public WolSwitch(int id, String ipAddress, String macAddress) {
        this(id, ipAddress, macAddress, PlatformEnum.WINDOWS);
    }

    /**
     * @param id         HomeKit Accessory Id
     * @param ipAddress  ipAddress
     * @param macAddress macAddress
     * @param platform   platform
     */
    public WolSwitch(int id, String ipAddress, String macAddress, PlatformEnum platform) {
        this(id, ipAddress, macAddress, platform, TimeUnit.SECONDS, 30);
    }

    /**
     * @param id         HomeKit Accessory Id
     * @param ipAddress  ipAddress
     * @param macAddress macAddress
     * @param platform   platform
     * @param timeUnit   timeUnit
     * @param timeOut    timeOut
     */
    public WolSwitch(int id, String ipAddress, String macAddress, PlatformEnum platform, TimeUnit timeUnit, long timeOut) {
        this(id, ipAddress, macAddress, platform, timeUnit, timeOut, 3);
    }

    /**
     * @param id         HomeKit Accessory Id
     * @param ipAddress  ipAddress
     * @param macAddress macAddress
     * @param platform   platform
     * @param timeUnit   timeUnit
     * @param timeOut    timeOut
     */
    public WolSwitch(int id, String ipAddress, String macAddress, PlatformEnum platform, TimeUnit timeUnit, long timeOut, int retry) {
        this(id, ipAddress, macAddress, platform, timeUnit, timeOut, retry, WOLNode.BROADCAST_IP, WOLNode.WOL_PORT);
    }

    public WolSwitch(int id, String ipAddress, String macAddress, PlatformEnum platform, TimeUnit timeUnit, long timeOut, int retry, String broadcastIp, int broadcastPort) {
        validate(id, macAddress);
        this.id = id;
        this.ipAddress = ipAddress;
        this.macAddress = macAddress;
        this.platform = platform;
        this.timeUnit = timeUnit;
        this.timeOut = timeOut;
        this.retry = retry;
        this.broadcastIp = broadcastIp;
        this.broadcastPort = broadcastPort;
    }

    private void validate(int id, String macAddress) {
        if (id == 1) {
            throw new IllegalArgumentException("this accessory id can not be 1.");
        }
        if (!MacAddressValidator.validate(macAddress)) {
            throw new IllegalArgumentException("mac address verification failed.");
        }
    }

    @Override
    public CompletableFuture<String> getName() {
        return CompletableFuture.completedFuture("电脑远程开关");
    }

    @Override
    public void identify() {
        log.info("this is a Computer WOL switch accessory");
    }

    @Override
    public CompletableFuture<String> getSerialNumber() {
        return CompletableFuture.completedFuture("Test SwitchSN");
    }

    @Override
    public CompletableFuture<String> getModel() {
        return CompletableFuture.completedFuture("Computer WOL Switch");
    }

    @Override
    public CompletableFuture<String> getManufacturer() {
        return CompletableFuture.completedFuture("XIMCloud Works");
    }

    @Override
    public CompletableFuture<String> getFirmwareRevision() {
        return CompletableFuture.completedFuture(WolSwitch.class.getPackage().getImplementationVersion());
    }

    @Override
    public CompletableFuture<String> getHardwareRevision() {
        return CompletableFuture.completedFuture("1.0.0");
    }

    /**
     * Retrieves the current binary state of the switch.
     *
     * @return a future that will contain the binary state
     */
    @Override
    public CompletableFuture<Boolean> getSwitchState() {
        return PingUtil.ping(ipAddress);
    }

    /**
     * Sets the binary state of the switch
     *
     * @param state the binary state to set
     * @return a future that completes when the change is made
     * @throws Exception when the change cannot be made
     */
    @Override
    public CompletableFuture<Void> setSwitchState(boolean state) throws Exception {
        if (Boolean.TRUE.equals(getSwitchState().join())) {
//            开机状态
            if (state) {
                log.warn("the computer is power on now,not need to wol");
            } else {
                ShutdownAndWakeUpUtil.shutdown(macAddress, ipAddress, platform, timeUnit.toSeconds(timeOut), retry);
                Optional.ofNullable(subscribeCallback).ifPresent(HomekitCharacteristicChangeCallback::changed);
            }
        } else {
//            关机状态
            if (state) {
                ShutdownAndWakeUpUtil.powerOn(macAddress, ipAddress, timeUnit.toSeconds(timeOut), retry, broadcastIp, broadcastPort);
                Optional.ofNullable(subscribeCallback).ifPresent(HomekitCharacteristicChangeCallback::changed);
            } else {
                log.warn("the computer is power off now,not need to power off");
            }

        }
        return CompletableFuture.completedFuture(null);
    }

    /**
     * Subscribes to changes in the binary state of the switch.
     *
     * @param callback the function to call when the state changes.
     */
    @Override
    public void subscribeSwitchState(HomekitCharacteristicChangeCallback callback) {
        this.subscribeCallback = callback;
    }

    /**
     * Unsubscribes from changes in the binary state of the switch.
     */
    @Override
    public void unsubscribeSwitchState() {
        this.subscribeCallback = null;
    }
}
