package cn.ximcloud.homekit.core.starter.core.auth.impl;

import cn.ximcloud.homekit.core.starter.core.auth.XIMCloudHomeKitAuthInfo;
import cn.ximcloud.homekit.core.starter.model.vo.HomeKitUserVo;
import io.github.hapjava.server.impl.HomekitServer;
import io.github.hapjava.server.impl.crypto.HAPSetupCodeUtils;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.io.*;
import java.math.BigInteger;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;


/**
 * XIMCloudHomeKitAuthInfo
 *
 * @author W9004844
 */
@Slf4j
public class SimpleDemoHomeKitAuthInfoImpl implements XIMCloudHomeKitAuthInfo {

    /**
     * pin
     */
    private final String pin;

    /**
     * mac
     */
    private final String mac;

    /**
     * salt
     */
    private final BigInteger salt;

    /**
     * privateKey
     */
    private final byte[] privateKey;

    /**
     * setupId
     */
    private final String setupId;

    /**
     * userKeyMap
     */
    private Map<String, byte[]> userMap = new ConcurrentHashMap<>();

    /**
     * authFile
     */
    private final transient File authFile = new File("auth-state.bin");

    @SneakyThrows
    public SimpleDemoHomeKitAuthInfoImpl() {
        if (authFile.exists()) {
            try (ObjectInputStream objectInputStream = new ObjectInputStream(new FileInputStream(authFile))) {
                log.debug("Using persisted auth");

                SimpleDemoHomeKitAuthInfoImpl simpleDemoHomeKitAuthInfoImpl = (SimpleDemoHomeKitAuthInfoImpl) objectInputStream.readObject();
                this.pin = simpleDemoHomeKitAuthInfoImpl.pin;
                this.mac = simpleDemoHomeKitAuthInfoImpl.mac;
                this.salt = simpleDemoHomeKitAuthInfoImpl.salt;
                this.privateKey = simpleDemoHomeKitAuthInfoImpl.privateKey;
                this.userMap = simpleDemoHomeKitAuthInfoImpl.userMap;
                this.setupId = simpleDemoHomeKitAuthInfoImpl.setupId;
            }
        } else {
            pin = HomekitServer.generatePin();
            mac = HomekitServer.generateMac();
            salt = HomekitServer.generateSalt();
            privateKey = HomekitServer.generateKey();
            setupId = HAPSetupCodeUtils.generateSetupId();
            notifyChanged();
        }
        log.info("The PIN for pairing is {}", pin);
    }

    private void notifyChanged() {
        try (FileOutputStream fileOutputStream = new FileOutputStream(authFile);
             ObjectOutputStream objectOutputStream = new ObjectOutputStream(fileOutputStream)) {
            objectOutputStream.writeObject(this);
            objectOutputStream.flush();
        } catch (IOException e) {
            log.error("write to file error", e);
        }
    }

    /**
     * A pin code used for pairing the device. This pin will be required within iOS in order to
     * complete pairing. The numbers cannot be sequential and should not have a repeating pattern.
     *
     * @return the pin code, in the form ###-##-###
     */
    @Override
    public String getPin() {
        return pin;
    }

    /**
     * A unique MAC address to be advertised with the HomeKit information. This does not have to be
     * the MAC address of the network interface. You can generate this using {@link
     * HomekitServer#generateMac()}.
     *
     * @return the unique MAC address.
     */
    @Override
    public String getMac() {
        return mac;
    }

    /**
     * The Salt that will be used when hashing the pin code to send to the client. You should generate
     * this using {@link HomekitServer#generateSalt()}.
     *
     * @return the Salt.
     */
    @Override
    public BigInteger getSalt() {
        return salt;
    }

    /**
     * The private key used by the server during pairing and message encryption. You should generate
     * this using {@link HomekitServer#generateKey()}
     *
     * @return the private key.
     */
    @Override
    public byte[] getPrivateKey() {
        return privateKey;
    }

    /**
     * A setup Id used for pairing the device using QR Code. It can be any alphanumeric combination of
     * the length of 4.
     *
     * @return setup id
     */
    @Override
    public String getSetupId() {
        return setupId;
    }

    @Override
    public void createUser(String username, byte[] publicKey) {
        if (!userMap.containsKey(username)) {
            userMap.putIfAbsent(username, publicKey);
            log.info("Added user for {}", username);
        } else {
            log.warn("Already have this user for {}", username);
        }
        notifyChanged();
    }

    @Override
    public void removeUser(String userName) {
        if (userMap.containsKey(userName)) {
            userMap.remove(userName);
            log.info("Removed pairing for {}", userName);
        } else {
            log.warn("the user {} not found", userName);
        }
        notifyChanged();
    }

    @Override
    public byte[] getUserPublicKey(String userName) {
        if (userMap.containsKey(userName)) {
            return userMap.get(userName);
        } else {
            log.warn("the user {} not found", userName);
            return new byte[0];
        }
    }

    /**
     * Called to check if a user has been created. The homekit accessory advertises whether the
     * accessory has already been paired. At this time, it's unclear whether multiple users can be
     * created, however it is known that advertising as unpaired will break in iOS 9. The default
     * value has been provided to maintain API compatibility for implementations targeting iOS 8.
     *
     * @return whether a user has been created and stored
     */
    @Override
    public boolean hasUser() {
        return !userMap.isEmpty();
    }

    @Override
    public List<HomeKitUserVo> getUserList() {
        return userMap.entrySet().stream()
                .map(entity -> HomeKitUserVo.builder().name(entity.getKey()).publicKey(entity.getValue()).build())
                .collect(Collectors.toList());
    }

    @Override
    public boolean hasUser(String userName) {
        return getUserPublicKey(userName).length == 0;
    }
}
