package cn.ximcloud.homekit.core.starter.core.bootstrap;

import cn.ximcloud.homekit.core.starter.autoconfig.HomeKitDataAutoConfiguration;
import cn.ximcloud.homekit.core.starter.autoconfig.HomeKitServiceAutoConfiguration;
import cn.ximcloud.homekit.core.starter.autoconfig.properties.HomeKitProperties;
import cn.ximcloud.homekit.core.starter.core.service.HomeKitService;
import com.google.common.collect.Maps;
import io.github.hapjava.accessories.HomekitAccessory;
import io.github.hapjava.server.impl.HomekitRoot;
import io.github.hapjava.server.impl.HomekitServer;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.context.annotation.Configuration;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import static cn.ximcloud.homekit.core.starter.util.common.QrToConsole.printQr;

/**
 * @author wizard
 */
@Slf4j
@Service
@Configuration
@AutoConfigureAfter(value = {HomeKitDataAutoConfiguration.class, HomeKitServiceAutoConfiguration.class})
public class HomeKitBootStrapServer implements InitializingBean, DisposableBean {

    private final HomeKitProperties homeKitProperties;

    private final HomeKitService homeKitService;

    private volatile boolean flag = false;

    private HomekitRoot homekitRoot;

    /**
     * 加载的HomeKit accessory
     */
    private Map<Integer, HomekitAccessory> accessoryCache;

    /**
     * 当前正在运行的 HomeKit accessory
     */
    private Map<Integer, HomekitAccessory> currentRunningHomeKitAccessoriesCache;

    private static final String UN_SET = "unset";

    @SuppressWarnings("all")
    public HomeKitBootStrapServer(HomeKitProperties homeKitProperties,
                                  HomeKitService homeKitService) {
        this.homeKitProperties = homeKitProperties;
        this.homeKitService = homeKitService;
    }

    /**
     * Invoked by the containing {@code BeanFactory} on destruction of a bean.
     */
    @Override
    public void destroy() {
        releaseAllAccessories();
        stopServer();
    }

    /**
     * Invoked by the containing {@code BeanFactory} after it has set all bean properties
     * and satisfied {@link BeanFactoryAware}, {@code ApplicationContextAware} etc.
     * <p>This method allows the bean instance to perform validation of its overall
     * configuration and final initialization when all bean properties have been set.
     */
    @Override
    public void afterPropertiesSet() throws IOException {
        log.debug("generate HomeKit Server to Stand by");
        homekitRoot = new HomekitServer(getInetAddress(), homeKitProperties.getPort())
                .createBridge(
                        homeKitService.getAuthInfo(),
                        homeKitProperties.getLabel(),
                        homeKitProperties.getManufacturer(),
                        homeKitProperties.getModel(),
                        homeKitProperties.getSerialNumber(),
                        homeKitProperties.getFirmwareRevision(),
                        homeKitProperties.getHardwareRevision());
        autoScanCheckAndAddAccessories();
        if (homeKitProperties.isAutoStart()) {
            log.debug("auto starting homeKit Server");
            homekitRoot.start();
            currentRunningHomeKitAccessoriesCache = Maps.newHashMap(accessoryCache);
            flag = true;
            log.debug("homeKit Server is running now");
        }
        printQr(homeKitService.getAuthInfo());
    }

    public InetAddress getInetAddress() {
        return Optional.ofNullable(homeKitProperties.getIpAddress()).map(ipAddress -> {
            try {
                return InetAddress.getByName(homeKitProperties.getIpAddress());
            } catch (UnknownHostException e) {
                throw new IllegalArgumentException();
            }
        }).orElseGet(() -> {
            try {
                return InetAddress.getLocalHost();
            } catch (UnknownHostException e) {
                throw new IllegalArgumentException();
            }
        });
    }

    public boolean isRunning() {
        return flag;
    }

    @SneakyThrows
    public synchronized void startServer() {
        if (!flag) {
            log.debug("starting HomeKit server");
            homekitRoot = new HomekitServer(homeKitProperties.getPort())
                    .createBridge(
                            homeKitService.getAuthInfo(),
                            homeKitProperties.getLabel(),
                            homeKitProperties.getManufacturer(),
                            homeKitProperties.getModel(),
                            homeKitProperties.getSerialNumber(),
                            homeKitProperties.getFirmwareRevision(),
                            homeKitProperties.getHardwareRevision());
            currentRunningHomeKitAccessoriesCache.values().forEach(homekitRoot::addAccessory);
            currentRunningHomeKitAccessoriesCache.values().forEach(HomeKitBootStrapServer::printAccessoryLog);
            homekitRoot.start();
            printQr(homeKitService.getAuthInfo());
            flag = true;
            log.debug("HomeKit server started successful");
        } else {
            log.warn("homeKit is running now ,can not restart");
        }
    }

    @SneakyThrows
    public void refreshAuthInfo() {
        homekitRoot.refreshAuthInfo();
    }

    public synchronized void stopServer() {
        currentRunningHomeKitAccessoriesCache.values().forEach(accessory -> homekitRoot.removeAccessory(accessory));
        currentRunningHomeKitAccessoriesCache.clear();
        log.debug("stopping HomeKit server");
        homekitRoot.stop();
        flag = false;
        log.debug("HomeKit server stopped successful now");
    }


    public List<HomekitAccessory> getCurrentLoadedHomeKitAccessories() {
        return Collections.unmodifiableList(new ArrayList<>(accessoryCache.values()));
    }

    public List<HomekitAccessory> getCurrentRunningHomeKitAccessories() {
        return Collections.unmodifiableList(new ArrayList<>(currentRunningHomeKitAccessoriesCache.values()));
    }


    /**
     * 检查并添加配件
     */
    private void autoScanCheckAndAddAccessories() {
        List<HomekitAccessory> homekitAccessories = homeKitService.getAccessories()
                .stream()
                .sorted(Comparator.comparing(HomekitAccessory::getId))
                .filter(accessory -> {
                    if (accessory.getId() == 1) {
                        log.warn("accessory [{}] has a warn id number 1", accessory);
                        return false;
                    }
                    return true;
                }).collect(Collectors.toList());
        cacheToAccessory(homekitAccessories);
    }

    private static void printAccessoryLog(HomekitAccessory accessory) {
        log.info("add accessory [Id:{},Name:{},SerialNumber:{},Model:{},Manufacturer:{},FirmwareRevision:{}]",
                accessory.getId(),
                accessory.getName().join(),
                Optional.ofNullable(accessory.getSerialNumber()).map(CompletableFuture::join).orElse(UN_SET),
                Optional.ofNullable(accessory.getModel()).map(CompletableFuture::join).orElse(UN_SET),
                Optional.ofNullable(accessory.getManufacturer()).map(CompletableFuture::join).orElse(UN_SET),
                Optional.ofNullable(accessory.getFirmwareRevision()).map(CompletableFuture::join).orElse(UN_SET)
        );
    }

    private void releaseAllAccessories() {
        currentRunningHomeKitAccessoriesCache.values().forEach(accessory -> homekitRoot.removeAccessory(accessory));
        currentRunningHomeKitAccessoriesCache.clear();
        accessoryCache.clear();
    }

    private void cacheToAccessory(List<HomekitAccessory> homekitAccessories) {
        accessoryCache = homekitAccessories.stream().collect(Collectors.toMap(HomekitAccessory::getId, x -> x));
        accessoryCache.values().forEach(HomeKitBootStrapServer::printAccessoryLog);
        accessoryCache.values().forEach(homekitRoot::addAccessory);
    }

    public void startAccessory(Integer id) {
        HomekitAccessory homekitAccessory = accessoryCache.get(id);
        homekitRoot.addAccessory(homekitAccessory);
        currentRunningHomeKitAccessoriesCache.put(id, homekitAccessory);
    }

    public void stopAccessory(Integer id) {
        HomekitAccessory homekitAccessory = currentRunningHomeKitAccessoriesCache.get(id);
        homekitRoot.removeAccessory(homekitAccessory);
        currentRunningHomeKitAccessoriesCache.remove(id);
    }
}
