package cn.ximcloud.homekit.core.starter.core.service.impl;

import cn.ximcloud.homekit.core.starter.autoconfig.properties.HomeKitDataSourceProperties;
import cn.ximcloud.homekit.core.starter.autoconfig.properties.HomeKitProperties;
import cn.ximcloud.homekit.core.starter.constants.HomeKitAccessoryTypeEnum;
import cn.ximcloud.homekit.core.starter.core.accessories.demo.DemoHomekitAccessoryBuilder;
import cn.ximcloud.homekit.core.starter.core.auth.XIMCloudHomeKitAuthInfo;
import cn.ximcloud.homekit.core.starter.core.auth.impl.BaseOnDataBaseHomeKitAuthInfoImpl;
import cn.ximcloud.homekit.core.starter.core.service.HomeKitService;
import cn.ximcloud.homekit.core.starter.entity.metadata.HomeKitAccessoryEntity;
import cn.ximcloud.homekit.core.starter.entity.metadata.HomeKitAccessoryMethodEntity;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAccessoryConfigRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAccessoryRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAuthInfoRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitUserRepository;
import cn.ximcloud.homekit.core.starter.util.homekit.HomeKitAccessoryScanner;
import cn.ximcloud.homekit.core.starter.util.homekit.HomekitAccessoryCoverUtil;
import io.github.hapjava.accessories.HomekitAccessory;
import lombok.extern.slf4j.Slf4j;
import org.reflections.ReflectionUtils;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.Method;
import java.util.*;

import static cn.ximcloud.homekit.core.starter.core.service.impl.SimpleDemoHomeKitServiceImpl.DEMO_ACCESSORY_PACKAGE_PATH;
import static cn.ximcloud.homekit.core.starter.util.homekit.HomekitAccessoryCoverUtil.coverToHomeKitAccessoryMethodEntity;


/**
 * @author jecessewen
 */
@Slf4j
public class BaseOnDataBaseHomeKitServiceImpl implements HomeKitService {

    private final HomeKitAccessoryConfigRepository homeKitAccessoryConfigRepository;

    private final HomeKitProperties homeKitProperties;

    private final HomeKitAuthInfoRepository homeKitAuthInfoRepository;

    private final HomeKitUserRepository homeKitUserRepository;

    private final HomeKitDataSourceProperties homeKitDataSourceProperties;

    private final HomeKitAccessoryRepository homeKitAccessoryRepository;

    private final DemoHomekitAccessoryBuilder demoHomekitAccessoryBuilder;

    private final XIMCloudHomeKitAuthInfo homeKitAuthInfo;

    private List<HomekitAccessory> accessories;

    public BaseOnDataBaseHomeKitServiceImpl(HomeKitAuthInfoRepository homeKitAuthInfoRepository,
                                            HomeKitUserRepository homeKitUserRepository,
                                            HomeKitProperties homeKitProperties,
                                            HomeKitAccessoryConfigRepository homeKitAccessoryConfigRepository,
                                            HomeKitDataSourceProperties homeKitDataSourceProperties,
                                            HomeKitAccessoryRepository homeKitAccessoryRepository,
                                            DemoHomekitAccessoryBuilder demoHomekitAccessoryBuilder) {
        this.homeKitAuthInfoRepository = homeKitAuthInfoRepository;
        this.homeKitUserRepository = homeKitUserRepository;
        this.homeKitProperties = homeKitProperties;
        this.homeKitAccessoryConfigRepository = homeKitAccessoryConfigRepository;
        this.homeKitDataSourceProperties = homeKitDataSourceProperties;
        this.homeKitAccessoryRepository = homeKitAccessoryRepository;
        this.demoHomekitAccessoryBuilder = demoHomekitAccessoryBuilder;
        this.homeKitAuthInfo = new BaseOnDataBaseHomeKitAuthInfoImpl(homeKitAuthInfoRepository, homeKitUserRepository);
        handlerHomeKitAccessoryMetaData();
    }

    {
        log.info("generate base on database homekit service");
    }

    /**
     * 获取HomeKit 配置
     *
     * @return MockAuthInfo
     */
    @Override
    public XIMCloudHomeKitAuthInfo getAuthInfo() {
        return homeKitAuthInfo;
    }


    /**
     * 设备列表
     *
     * @return List<HomeKitAccessory>
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<HomekitAccessory> getAccessories() {
        return Optional.ofNullable(accessories).orElseGet(() -> {
            accessories = new ArrayList<>();
            addAccessories(accessories);
            return accessories;
        });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addAccessories(List<HomekitAccessory> accessories) {
        addDefaultAccessories(accessories);
    }

    @Transactional(rollbackFor = Exception.class)
    public void handlerHomeKitAccessoryMetaData() {
        HomeKitAccessoryScanner.scanHomeKitAccessory()
                .forEach(homeKitAccessoryClass -> extracted(homeKitAccessoryClass, HomeKitAccessoryTypeEnum.ACCESSORY));

        HomeKitAccessoryScanner.scanHomeKitAccessoryOptionalCharacteristic()
                .forEach(optionalCharacteristic -> extracted(optionalCharacteristic, HomeKitAccessoryTypeEnum.OPTIONAL_CHARACTERISTIC));
    }

    private void extracted(Class<?> homeKitAccessoryClass, HomeKitAccessoryTypeEnum type) {
        if (!homeKitAccessoryRepository.findByClazz(homeKitAccessoryClass).isPresent()) {
            HomeKitAccessoryEntity entity = new HomeKitAccessoryEntity();
            entity.setClazz(homeKitAccessoryClass);
            entity.setType(type);
            List<HomeKitAccessoryMethodEntity> methodList = new ArrayList<>();
            @SuppressWarnings("unchecked")
            Set<Method> allMethods = ReflectionUtils.getAllMethods(homeKitAccessoryClass);
            allMethods.forEach(method -> {
                HomeKitAccessoryMethodEntity homeKitAccessoryMethodEntity = coverToHomeKitAccessoryMethodEntity(method);
                homeKitAccessoryMethodEntity.setHomeKitAccessory(entity);
                methodList.add(homeKitAccessoryMethodEntity);
            });
            entity.setHomekitAccessoryMethodList(methodList);
            homeKitAccessoryRepository.save(entity);
        }
    }

    /**
     * 添加默认HomeKitAccessory配件
     *
     * @param accessories accessories
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addDefaultAccessories(List<HomekitAccessory> accessories) {
//        添加demo配件
        generateDemoAccessoriesWithCondition(accessories);

//        添加数据库中的配件
        generateFromDataBaseAccessories(accessories);

//        包扫描路径的配件
        generateBaseScanPackagePathAccessories(accessories);
    }

    @Override
    public void generateBaseScanPackagePathAccessories(List<HomekitAccessory> accessories) {
        Optional.ofNullable(homeKitProperties.getBaseScanAccessoriesPackagePath())
                .ifPresent(paths ->
                        Arrays.stream(paths)
                                .forEach(path -> accessories.addAll(HomeKitAccessoryScanner.scanHomeKitAccessoryAndBuildInstance(path))));
    }

    /**
     * 添加demo配件
     *
     * @param accessories accessories
     */
    private void generateDemoAccessoriesWithCondition(List<HomekitAccessory> accessories) {
        if (homeKitProperties.isAddDemo()) {
            accessories.addAll(HomeKitAccessoryScanner.scanHomeKitAccessoryAndBuildInstance(DEMO_ACCESSORY_PACKAGE_PATH));
        }
    }

    /**
     * 添加数据库中的配件
     *
     * @param accessories
     */
    @Transactional(rollbackFor = Exception.class)
    public void generateFromDataBaseAccessories(List<HomekitAccessory> accessories) {
        if (homeKitProperties.isAddDemo() && homeKitAccessoryConfigRepository.count() == 0) {
//            若数据库中无配件，保存一个配件至数据库
            homeKitAccessoryConfigRepository.save(demoHomekitAccessoryBuilder.buildDemoHomeKitAccessory());
        }
        homeKitAccessoryConfigRepository.findAll()
                .forEach(homeKitAccessoryConfigEntity ->
                        accessories.add(HomekitAccessoryCoverUtil.coverToHomekitAccessory(homeKitAccessoryConfigEntity)));
    }


}
