package cn.ximcloud.homekit.core.starter.util.homekit;

import cn.ximcloud.homekit.core.starter.util.annotation.HomeKit;
import com.google.common.collect.Maps;
import io.github.hapjava.accessories.HomekitAccessory;
import io.github.hapjava.accessories.optionalcharacteristic.AccessoryWithName;
import lombok.extern.slf4j.Slf4j;
import org.reflections.ReflectionUtils;
import org.reflections.Reflections;
import org.reflections.scanners.MethodAnnotationsScanner;
import org.reflections.scanners.SubTypesScanner;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author jecessewen
 */
@Slf4j
public class HomeKitAccessoryScanner {

    private HomeKitAccessoryScanner() {
    }

    private static final String HOMEKIT_ACCESSORY_PACKAGE_PATH = HomekitAccessory.class.getPackage().getName();

    private static final String HOMEKIT_ACCESSORY_OPTIONAL_CHARACTERISTIC_PACKAGE_PATH = AccessoryWithName.class.getPackage().getName();


    public static Set<Class<? extends HomekitAccessory>> scanHomeKitAccessory() {
        return new Reflections(HOMEKIT_ACCESSORY_PACKAGE_PATH).getSubTypesOf(HomekitAccessory.class);
    }

    public static List<Class<?>> scanHomeKitAccessoryOptionalCharacteristic() {
        return new Reflections(HOMEKIT_ACCESSORY_OPTIONAL_CHARACTERISTIC_PACKAGE_PATH, Collections.singletonList(new SubTypesScanner(false))).getAllTypes()
                .stream()
                .map(ReflectionUtils::forName)
                .collect(Collectors.toList());
    }

    public static Collection<? extends HomekitAccessory> scanHomeKitAccessoryAndBuildInstance(String accessoryPackagePath) {
        return scanHomeKitAccessoryAndBuildInstance(accessoryPackagePath, true);
    }

    public static Collection<? extends HomekitAccessory> scanHomeKitAccessoryAndBuildInstance(String accessoryPackagePath, boolean isAnnotationBinding) {
        Reflections reflections = new Reflections(accessoryPackagePath, Arrays.asList(new MethodAnnotationsScanner(), new SubTypesScanner(false)));
        Map<Integer, HomekitAccessory> classCollect = reflections.getSubTypesOf(HomekitAccessory.class)
                .stream()
                .filter(accessory -> !accessory.isInterface())
                .filter(accessory -> isAnnotationBinding && accessory.isAnnotationPresent(HomeKit.class))
                .filter(accessory -> {
                    try {
                        return Modifier.isPublic(accessory.getDeclaredConstructor().getModifiers());
                    } catch (NoSuchMethodException e) {
                        log.error("scan accessory error", e);
                        return false;
                    }
                })
                .map(accessory -> {
                    try {
                        return accessory.getConstructor().newInstance();
                    } catch (InstantiationException | IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                        log.error("scan accessory error", e);
                        return null;
                    }
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toConcurrentMap(HomekitAccessory::getId, item -> item));

        Map<Integer, ? extends HomekitAccessory> methodCollect;
        try {
            methodCollect = reflections.getMethodsAnnotatedWith(HomeKit.class)
                    .stream()
                    .filter(method -> HomekitAccessory.class.isAssignableFrom(method.getReturnType()))
                    .filter(method -> Modifier.isPublic(method.getModifiers()))
                    .filter(method -> method.getParameterCount() == 0)
                    .map(method -> {
                        if (Modifier.isStatic(method.getModifiers())) {
                            try {
                                return (HomekitAccessory) method.invoke(method.getDeclaringClass());
                            } catch (IllegalAccessException | InvocationTargetException e) {
                                log.error("generate homekit accessory error",e);
                                return null;
                            }
                        } else {
                            try {
                                return (HomekitAccessory) method.invoke(method.getDeclaringClass().getConstructor().newInstance());
                            } catch (NoSuchMethodException | IllegalAccessException | InstantiationException | InvocationTargetException e) {
                                log.error("generate homekit accessory error",e);
                                return null;
                            }
                        }
                    })
                    .filter(Objects::nonNull)
                    .collect(Collectors.toConcurrentMap(HomekitAccessory::getId, item -> item));
        } catch (Exception e) {
            log.error("scan accessory within method error,please check method is add @HomeKit or have any method");
            methodCollect = Maps.newHashMap();
        }
        classCollect.putAll(methodCollect);
        return classCollect.values();
    }

}
