package cn.ximcloud.homekit.core.starter.core.adapter;

import cn.ximcloud.homekit.core.starter.config.HomeKitDataSourceConfig;
import cn.ximcloud.homekit.core.starter.constants.DataSourceConstant;
import cn.ximcloud.homekit.core.starter.core.condition.EmbeddedDataBaseCondition;
import cn.ximcloud.homekit.core.starter.core.condition.SimpleDemoHomeKitCondition;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.jpa.repository.config.EnableJpaRepositories;
import org.springframework.stereotype.Component;

import javax.sql.DataSource;
import java.util.Optional;

@Slf4j
@Component
@Configuration
@EnableJpaRepositories
@ComponentScan("cn.ximcloud.homekit.core.starter")
@AutoConfigureAfter(HomeKitDataSourceConfig.class)
@AutoConfigureBefore(value = {DataSource.class, HomeKitServiceAdapter.class})
public class HomeKitDataSourceAdapter {

    static {
        log.info("init datasource adapter");
    }

    private final HomeKitDataSourceConfig homeKitDataSourceConfig;

    public HomeKitDataSourceAdapter(HomeKitDataSourceConfig homeKitDataSourceConfig) {
        this.homeKitDataSourceConfig = homeKitDataSourceConfig;
    }

    @Bean
    @ConditionalOnMissingBean(DataSource.class)
    @Conditional(SimpleDemoHomeKitCondition.class)
    public DataSource generateUnUsedDataSource() {
        log.info("generate unused DataSource");
        HikariConfig hikariConfig = new HikariConfig();
        hikariConfig.setDriverClassName("org.h2.Driver");
        hikariConfig.setJdbcUrl("jdbc:h2:mem:unused");
        hikariConfig.setUsername("sa");
        hikariConfig.setPassword("sa");
        return new HikariDataSource(hikariConfig);
    }

    @Bean
    @ConditionalOnMissingBean(DataSource.class)
    @Conditional(EmbeddedDataBaseCondition.class)
    public DataSource generateEmbeddedDataSource() {
        log.info("generate Embedded DataSource");
        HikariConfig hikariConfig = new HikariConfig();
        hikariConfig.setDriverClassName("org.h2.Driver");
        hikariConfig.setJdbcUrl(generateJdbcUrl());
        hikariConfig.setUsername("sa");
        hikariConfig.setPassword("sa");
        return new HikariDataSource(hikariConfig);
    }

    private String generateJdbcUrl() {
        if (DataSourceConstant.EMBEDDED.equals(homeKitDataSourceConfig.getType())) {
            return "jdbc:h2:mem:ximcloud_homekit";
        } else {
            return "jdbc:h2:file:".concat(Optional.ofNullable(homeKitDataSourceConfig.getFile()).orElse("../db/ximcloud_homekit"));
        }
    }

}
