/*
 * Copyright (c) 2019 huipei.x
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.xphsc.xpack;



import java.io.File;
import java.io.IOException;

import cn.xphsc.xpack.domain.*;
import cn.xphsc.xpack.utils.Logger;
import cn.xphsc.xpack.utils.Times;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.StringUtils;

/**
 * @author <a href="xiongpeih@163.com">huipei.x</a>
 * @description:
 * @since 1.0.0
 */
public abstract class AbstractMojoBuilder extends AbstractMojo {


    private static final String START ="-------------------------- Xpack start packing... -------------------------\n" +

            "___    ___                   __    \n" +
            "\\   \\/  /__________    ____ |  | __\n" +
            " \\     /\\____ \\__  \\ _/ ___\\|  |/ /\n" +
            " /     \\|  |_> > __ \\\\  \\___|    < \n" +
            "/___/\\  \\   __(____  /\\___  >__|_ \\\n" +
            "      \\_/__|       \\/     \\/     \\/";

    /**
     * The folder name constant used to store xpack packaging.
     */
    static final String HOME_DIR_NAME = "xpack";

    /**
     * JDK8 Mirror Name Constant.
     */
    private static final String JDK8_IMAGE = "jdk:8-jdk-alpine";

    /**
     * Default data volume string array constants that need to be mounted.
     */
    private static final String[] DEFAULT_VOLUMES = new String[] {"/tmp", "logs"};

    /**
     *File objects in the target directory of the Maven runtime.
     */
    @Parameter(defaultValue = "${project.build.directory}", required = true)
    private File targetDir;

    /**
     * Maven `groupId'.
     */
    @Parameter(defaultValue = "${project.groupId}", required = true)
    private String groupId;

    /**
     * The `artifactId'of the Maven project.
     */
    @Parameter(defaultValue = "${project.artifactId}", required = true)
    private String artifactId;

    /**
     * Maven `292563;'version.
     */
    @Parameter(defaultValue = "${project.version}", required = true)
    private String version;


    /**
     * The `finalName'in the Maven project pom. xml. If this item is set in Maven, the packaged name prefix will be this value.
     * Otherwise, the name prefix of the package Maven defaults to `artifactId + version'.
     */
    @Parameter(defaultValue = "${project.build.finalName}", required = true)
    private String finalName;
    @Parameter(defaultValue="false",property = "nameVersion")
    private boolean nameVersion;
    /**
     * The version `description'in the Maven project pom.xml.
     */
    @Parameter(defaultValue = "${project.description}")
    private String description;

    /**
     * JVM-related parameters when running Java programs.
     */
    @Parameter(property = "vmOptions")
    private String vmOptions;

    /**
     * Parameters that may be required by the program itself when running a Java program.
     */
    @Parameter(property = "programArgs")
    private String programArgs;

    /**
     * Configuration file paths required to run SpringBook programs can be relative or absolute paths, and multiple paths can be filled in.
     */
    @Parameter(property = "configFiles")
    private String[] configFiles;

    /**
     * An array of supported packaged platforms, if not or empty, is considered to support all platforms.
     */
    @Parameter(property = "platforms")
    protected String[] platforms;

    /**
     * Whether to skip exceptions or errors during execution, if true, skip straight without throwing exceptions, otherwise throw exceptions, the default value is default, the default processing will be compromised.
     */
    @Parameter(property = "skipError")
    private String skipError;

    /**
     * Construct parameters related to Windows publishing packages.
     */
    @Parameter(property = "windows")
    private Windows windows;

    /**
     * Construct parameters related to Linux publishing packages.
     */
    @Parameter(property = "linux")
    private Linux linux;

    /**
     * Construct parameters related to Docker publishing packages.
     */
    @Parameter(property = "docker")
    private Docker docker;

    /**
     * Files or directories that need to be excluded (i.e., not generated).
     */
    @Parameter(property = "excludeFiles")
    private String[] excludeFiles;

    /**
     * Copy related resources to custom configuration parameters in each platform package.
     */
    @Parameter(property = "copyResources")
    private CopyResource[] copyResources;



    /**
     * Perform whatever build-process behavior this <code>Mojo</code> implements.<br>
     * This is the main trigger for the <code>Mojo</code> inside the <code>Maven</code> system, and allows
     * the <code>Mojo</code> to communicate errors.
     */
    @Override
    public void execute() {
        Logger.initSetLog(super.getLog());
        Logger.info(START);
        final long start = System.nanoTime();

        this.exec();

        Logger.info("------------- xpack has been packaged to end. [costs: "
                + Times.convertTime(System.nanoTime() - start) + "] -------------\n");
    }


    protected abstract void exec();

    /**
     * The PackInfo object instance is constructed to facilitate the transfer and reuse of this object attribute.
     */
    protected PackInfo buildPackInfo() {
        PackInfo packInfo = new PackInfo()
                .setTargetDir(this.targetDir)
                .setHomeDir(this.createHomeDir())
                .setArtifactId(this.artifactId)
                .setVersion(this.version)
                .setNameVersion(this.nameVersion)
                .setName(this.finalName)
                .setDescription(this.description)
                .setVmOptions(this.vmOptions)
                .setProgramArgs(this.programArgs)
                .setConfigFiles(this.configFiles)
                .setSkipError(SkipErrorEnum.of(this.skipError))
                .setWindows(this.windows)
                .setLinux(this.linux)
                .setDocker(this.initDefaultDockerInfo())
                .setExcludeFiles(this.excludeFiles)
                .setCopyResources(this.copyResources);
        Logger.debug(packInfo.toString());
        return packInfo;
    }

    /**
     * Create the xpack home folder directory.
     *
     */
    private File createHomeDir() {
        File file = new File(this.targetDir + File.separator + HOME_DIR_NAME + File.separator);
        try {
            if (file.exists()) {
                FileUtils.cleanDirectory(file);
            } else {
                FileUtils.forceMkdir(file);
            }
        } catch (IOException e) {
            Logger.error("\n" +
                    "Failed to create xpack folder! Please check if any files are in use.! ", e);
        }
        return file;
    }

    /**
     *
     Default information related to building Docker images.
     *
     */
    private Docker initDefaultDockerInfo() {
        if (this.docker == null) {
            this.docker = new Docker();
        }

        if (StringUtils.isBlank(this.docker.getRepo())) {
            this.docker.setRepo(this.groupId);
        }
        if (StringUtils.isBlank(this.docker.getName())) {
            this.docker.setName(this.artifactId);
        }
        if (StringUtils.isBlank(this.docker.getTag())) {
            this.docker.setTag(this.version);
        }
        if (StringUtils.isBlank(this.docker.getFromImage())) {
            this.docker.setFromImage(JDK8_IMAGE);
        }
        if (ArrayUtils.isEmpty(this.docker.getVolumes())) {
            this.docker.setVolumes(DEFAULT_VOLUMES);
        }
        return this.docker;
    }

    /* getter and setter methods. */

    void setTargetDir(File targetDir) {
        this.targetDir = targetDir;
    }

    void setGroupId(String groupId) {
        this.groupId = groupId;
    }

    void setArtifactId(String artifactId) {
        this.artifactId = artifactId;
    }

    void setVersion(String version) {
        this.version = version;
    }

    void setFinalName(String finalName) {
        this.finalName = finalName;
    }

    void setDescription(String description) {
        this.description = description;
    }

    public Docker getDocker() {
        return docker;
    }

}
