/*
 * Copyright (c) 2019 huipei.x
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.xphsc.xpack.domain;



import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang3.ArrayUtils;
import org.codehaus.plexus.util.StringUtils;

/**
 * @author <a href="xiongpeih@163.com">huipei.x</a>
 * @description:
 * @since 1.0.0
 */
public enum PlatformEnum {

    /**
     * windows.
     */
    WINDOWS("windows") {
        @Override
        public PackInfo mergeNewPackInfo(PackInfo packInfo) {
            return newBaseConfigPackInfo(packInfo, packInfo.getWindows());
        }
    },

    /**
     * linux.
     */
    LINUX("linux") {
        @Override
        public PackInfo mergeNewPackInfo(PackInfo packInfo) {
            return newBaseConfigPackInfo(packInfo, packInfo.getLinux());
        }
    },

    /**
     * docker.
     */
    DOCKER("docker") {
        @Override
        public PackInfo mergeNewPackInfo(PackInfo packInfo) {
            return newBaseConfigPackInfo(packInfo, packInfo.getDocker());
        }
    };

    /**
     * 属性值.
     */
    private String code;

    /**
     * 构造方法.
     *
     * @param code code值
     */
    PlatformEnum(String code) {
        this.code = code;
    }

    /**
     * 根据 platform 的字符串值转换为 PlatformEnum 的值.
     *
     * @param platform 平台字符串
     * @return PlatformEnum实例
     */
    public static PlatformEnum of(String platform) {
        for (PlatformEnum platformEnum : PlatformEnum.values()) {
            if (platformEnum.code.equals(platform)) {
                return platformEnum;
            }
        }
        return null;
    }

    /**
     * 获取所有平台的List集合.
     *
     * @return 集合
     */
    public static List<PlatformEnum> getPlatformList() {
        List<PlatformEnum> platformList = new ArrayList<>(PlatformEnum.values().length);
        Collections.addAll(platformList, PlatformEnum.values());
        return platformList;
    }

    /**
     * 创建一个新的 PackInfo 实例，且赋予了各个平台自己的一些配置信息数据.
     *
     * @param packInfo PackInfo
     * @param baseConfig baseConfig
     * @return PackInfo 实例
     */
    public static PackInfo newBaseConfigPackInfo(PackInfo packInfo, BaseConfigure baseConfig) {
        PackInfo newPackInfo = PackInfo.newCommonPackInfo(packInfo);
        if (StringUtils.isNotBlank(baseConfig.getVmOptions())) {
            newPackInfo.setVmOptions(baseConfig.getVmOptions());
        }
        if (StringUtils.isNotBlank(baseConfig.getProgramArgs())) {
            newPackInfo.setProgramArgs(baseConfig.getProgramArgs());
        }
        if (ArrayUtils.isNotEmpty(baseConfig.getConfigFiles())) {
            newPackInfo.setConfigFiles(ArrayUtils.addAll(packInfo.getConfigFiles(), baseConfig.getConfigFiles()));
        }
        if (ArrayUtils.isNotEmpty(baseConfig.getCopyResources())) {
            newPackInfo.setCopyResources(
                    ArrayUtils.addAll(packInfo.getCopyResources(), baseConfig.getCopyResources()));
        }
        if (ArrayUtils.isNotEmpty(baseConfig.getExcludeFiles())) {
            newPackInfo.setExcludeFiles(ArrayUtils.addAll(packInfo.getExcludeFiles(), baseConfig.getExcludeFiles()));
        }
        return newPackInfo;
    }

    /**
     * 根据 PackInfo 对象信息合并出适合各个平台自己的一个新的 PackInfo 对象，用于覆盖通用的配置信息.
     * <p>合并策略如下：</p>
     * <ul>
     *     <li>针对配置项只有一个值的情况，使用"覆盖"的方式来合并配置项，各平台自己的配置项优先级最高，为空时使用公用的配置项. </li>
     *     <li>针对配置项有多个值的情况，使用取"并集"的方式来合并配置项，各平台自己的配置项和公用的配置项取并集. </li>
     * </ul>
     *
     * @param packInfo PackInfo对象
     * @return PackInfo对象
     */
    public abstract PackInfo mergeNewPackInfo(PackInfo packInfo);

    /**
     * 获取该平台的 code 值.
     *
     * @return code
     */
    public String getCode() {
        return code;
    }

}
