/*
 * Copyright (c) 2019 huipei.x
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.xphsc.xpack.handler;



import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import cn.xphsc.xpack.domain.PackInfo;
import cn.xphsc.xpack.utils.Compress;
import cn.xphsc.xpack.utils.Logger;
import cn.xphsc.xpack.domain.PlatformEnum;
import cn.xphsc.xpack.domain.CopyResource;
import org.apache.commons.lang3.ArrayUtils;
import org.codehaus.plexus.DefaultPlexusContainer;
import org.codehaus.plexus.PlexusContainerException;
import org.codehaus.plexus.component.repository.exception.ComponentLookupException;
import org.codehaus.plexus.resource.ResourceManager;
import org.codehaus.plexus.resource.loader.FileResourceCreationException;
import org.codehaus.plexus.resource.loader.ResourceNotFoundException;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.StringUtils;

/**
 * @author <a href="xiongpeih@163.com">huipei.x</a>
 * @description:
 * @since 1.0.0
 */
public abstract class AbstractPackHandler implements PackHandler {

    private static final String HTTP = "http://";

    private static final String HTTPS = "https://";


    protected static final String BIN_DIR_NAME = "bin";

    /**
     *
     Resource Manager Object.
     */
    private ResourceManager resourceManager;

    /**
     * Packed Relevant Parameter Entity Objects.
     */
    protected PackInfo packInfo;

    /**
     * Home directory path in each platform.
     */
    protected String platformPath;

    /**
     * The bin directory path in the home directory of each platform.
     */
    protected String binPath;


    protected void createPlatformCommonDir(PlatformEnum platformEnum) {
        this.initPlatformPath(platformEnum);

        try {
            this.resourceManager = (ResourceManager) new DefaultPlexusContainer().lookup(ResourceManager.ROLE);

            FileUtils.mkdir(this.platformPath);

            this.copyJarAndConfigFile();
        } catch (PlexusContainerException | ComponentLookupException e) {
            Logger.error("创建【" + this.platformPath + "】目录或者复制相关的资源失败！请检查文件是否正在使用!", e);
        }
    }


    protected void createBaseDirs() {
        this.binPath = this.platformPath + File.separator + AbstractPackHandler.BIN_DIR_NAME + File.separator;
        try {
            FileUtils.forceMkdir(new File(binPath));
            FileUtils.forceMkdir(new File(this.platformPath + File.separator + "config"));
            FileUtils.forceMkdir(new File(this.platformPath + File.separator + "docs"));
            FileUtils.forceMkdir(new File(this.platformPath + File.separator + "logs"));
            FileUtils.forceMkdir(new File(this.platformPath + File.separator + "lib"));
        } catch (IOException e) {
            Logger.error("创建【" + this.platformPath + "】目录下的 bin、config、docs、logs 等目录失败！请检查文件是否正在使用!", e);
        }
    }


    private void copyJarAndConfigFile() {
        String jar = packInfo.getFullJarName();
        try {
            FileUtils.copyFileToDirectory(packInfo.getTargetDir().getAbsolutePath() + File.separator + jar,
                    platformPath);
            if(!packInfo.isNameVersion()){
                File sourceFile = new File(platformPath  + File.separator + jar);
                File targetFile = new File(platformPath + File.separator + packInfo.getArtifactId()+PackInfo.getJAR());
                sourceFile.renameTo(targetFile);
            }
            String[] configFiles = this.packInfo.getConfigFiles();
            if (ArrayUtils.isNotEmpty(configFiles)) {
                for (String configFile : configFiles) {
                    this.copyCustomResources(configFile, "config");
                }
            }
        } catch (IOException e) {
            Logger.error("复制【" + jar + "】到【" + platformPath + "】目录中失败！应该还没有打包此文件，"
                    + "异常信息为：" + e.getMessage());
        }
    }


    private void initPlatformPath(PlatformEnum platformEnum) {
        this.platformPath = packInfo.getHomeDir().getAbsolutePath() + File.separator + platformEnum.getCode();
    }


    protected void copyFiles(String source, String destination) {
        try {
            FileUtils.copyFile(this.resourceManager.getResourceAsFile(source),
                    new File(this.platformPath, destination));
        } catch (IOException | ResourceNotFoundException | FileResourceCreationException e) {
            Logger.error("复制默认资源到平台中出错！", e);
        }
    }


    protected Map<String, Object> buildBaseTemplateContextMap() {
        Map<String, Object> context = new HashMap<>(8);
        context.put("name", (packInfo.isNameVersion()?packInfo.getName():packInfo.getArtifactId()));
        context.put("jarName", (packInfo.isNameVersion()?packInfo.getFullJarName():packInfo.getArtifactId()+PackInfo.getJAR()));
        context.put("vmOptions", StringUtils.isBlank(packInfo.getVmOptions()) ? "" : packInfo.getVmOptions());
        context.put("programArgs", StringUtils.isBlank(packInfo.getProgramArgs()) ? "" : packInfo.getProgramArgs());
        return context;
    }


    protected boolean isRootPath(String filePath) {
        return StringUtils.isBlank(filePath) || ".".equals(filePath) || "/".equals(filePath);
    }


    private void copyCustomResources() {
        CopyResource[] copyResources = packInfo.getCopyResources();
        if (ArrayUtils.isEmpty(copyResources)) {
            return;
        }

        for (CopyResource copyResource : copyResources) {
            String fromPath = copyResource.getOriginalResource();
            try {
                this.copyCustomResources(fromPath, copyResource.getTargetResource());
            } catch (IOException e) {
                Logger.error("复制配置的自定义资源【" + fromPath + "】到各平台的包中出错！", e);
            }
        }
    }


    private void copyCustomResources(String originalResource, String targetResource) throws IOException {
        if (StringUtils.isBlank(originalResource)) {
            return;
        }

        if (originalResource.startsWith(HTTP) || originalResource.startsWith(HTTPS)) {
            String[] arr = originalResource.split("/");
            File dir = new File(this.platformPath + File.separator + targetResource);
            FileUtils.forceMkdir(dir);
            FileUtils.copyURLToFile(new URL(originalResource), new File(dir + File.separator + arr[arr.length - 1]));
        } else {
            File sourceFile = new File(originalResource);
            if (!sourceFile.exists()) {
                Logger.warn("【警告】需要复制的源资源文件【" + originalResource + "】不存在，请检查！");
                return;
            }

            // 如果 to 是空的或者 `.`、'/', 则表示复制到各平台包的根目录中，否则复制到对应的目录中即可.
            File toDir = new File(this.isRootPath(targetResource) ? this.platformPath : this.platformPath + File.separator + targetResource);

            // 如果需要复制的资源是目录，则直接复制该目录及其下的子目录到目标目录中.
            if (sourceFile.isDirectory()) {
                FileUtils.copyDirectoryStructure(sourceFile, toDir);
            } else {
                FileUtils.copyFileToDirectory(sourceFile, toDir);
            }
        }
    }

    /**
     * Exclude (delete) unnecessary files or directories.
     */
    private void excludeFiles() {
        String[] excludeFiles = packInfo.getExcludeFiles();
        if (ArrayUtils.isEmpty(excludeFiles)) {
            return;
        }

        for (String path : excludeFiles) {
            String filePath = this.platformPath + File.separator + path;
            File file = new File(filePath);
            if (!file.exists()) {
                Logger.warn("【警告】你配置的需要排除的资源【" + filePath + "】不存在，请检查！");
                continue;
            }

            try {
                FileUtils.forceDelete(file);
            } catch (IOException e) {
                Logger.error("【错误】删除配置的需要排除的资源【" + filePath + "】出错！", e);
            }
        }
    }

    /**
     *
     Some processing that needs to be done before compression.
     */
    private void handleBeforeCompress() {
        this.copyCustomResources();

        this.excludeFiles();
    }

    /**
     * Make tar.gz compression package under linux.
     */
    protected void compress(PlatformEnum platformEnum) {
        // Some common processing operations need to be done before compressing the platform folders.
        this.handleBeforeCompress();

        String platform = platformEnum.getCode();
        Logger.debug("正在制作 " + platform + " 下的部署压缩包...");
        try {
            switch (platformEnum) {
                case WINDOWS:
                    Compress.zip(platformPath, packInfo.getPackName() + ".zip");
                    break;
                case LINUX:
                    Compress.tarGz(platformPath, packInfo.getPackName() + ".tar.gz");
                    break;
                case DOCKER:
                    Compress.tarGz(platformPath, packInfo.getDockerPackName() + ".tar.gz");
                    break;
                default:
                    break;
            }
            FileUtils.forceDelete(platformPath);
            Logger.debug("已清除 " + platform + " 临时文件.");
        } catch (IOException e) {
            Logger.error("压缩并清除 " + platform + " 下部署的临时文件失败.", e);
        }
        Logger.info("制作 " + platform + " 下的部署压缩包完成.");
    }

}
