/*
 * Copyright (c) 2019 huipei.x
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.xphsc.xpack.handler;



import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import cn.xphsc.xpack.handler.impl.DockerPackHandler;
import cn.xphsc.xpack.handler.impl.LinuxPackHandler;
import cn.xphsc.xpack.handler.impl.WindowsPackHandler;
import cn.xphsc.xpack.domain.PlatformEnum;
import cn.xphsc.xpack.domain.PackInfo;
import cn.xphsc.xpack.utils.Logger;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * @author <a href="xiongpeih@163.com">huipei.x</a>
 * @description:
 * @since 1.0.0
 */
public class XpackContext {


    private static final Map<PlatformEnum, PackHandler> PACK_HANDLER_MAP = new LinkedHashMap<>(4);

    static {
        PACK_HANDLER_MAP.put(PlatformEnum.WINDOWS, new WindowsPackHandler());
        PACK_HANDLER_MAP.put(PlatformEnum.LINUX, new LinuxPackHandler());
        PACK_HANDLER_MAP.put(PlatformEnum.DOCKER, new DockerPackHandler());
    }

    /**
     * The method of packaging according to the relevant parameters of packaging.
     */
    public void pack(String[] platforms, final PackInfo packInfo) {
        List<PlatformEnum> platformEnums = this.convertPlatformList(platforms);

        // Initialize the object instance of thread pool and CountDownLatch.
        int count = platformEnums.size();
      final CountDownLatch countDownLatch = new CountDownLatch(count);
        ExecutorService executorService=  new ThreadPoolExecutor(count, count,
                0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new ThreadFactory() {
                    @Override
                    public Thread newThread(Runnable r) {
                        return new Thread(r, "xpack-thread");
                    }
                });


        /**
         *Traversing through various platforms, multi-threaded construction and packaging.
         */
        for (final PlatformEnum platformEnum : platformEnums) {
            executorService.submit(new Runnable() {
                @Override
                public void run() {
                    PACK_HANDLER_MAP.get(platformEnum).pack(platformEnum.mergeNewPackInfo(packInfo));
                    countDownLatch.countDown();
                }
            });
        }

        this.awaitAndShutdown(countDownLatch, executorService);
    }

    /**
     The platforms platform for transformation configuration is an enumeration set.
     */
    private List<PlatformEnum> convertPlatformList(String[] platforms) {
        if (ArrayUtils.isEmpty(platforms)) {
            return PlatformEnum.getPlatformList();
        }

        List<PlatformEnum> platformEnumList = new ArrayList<>(platforms.length);
        for (String platform : platforms) {
            if (StringUtils.isBlank(platform)) {
                Logger.warn("你配置的 xpack 平台为空！");
                continue;
            }

            PlatformEnum platformEnum = PlatformEnum.of(platform.trim().toLowerCase());
            if (platformEnum == null) {
                Logger.warn("你配置的 xpack 平台 " + platform + " 不存在！");
            } else {
                platformEnumList.add(platformEnum);
            }
        }
        return platformEnumList;
    }


    private void awaitAndShutdown(CountDownLatch countDownLatch, ExecutorService executorService) {
        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            Logger.error("在多线程下等待 xpack 构建打包结束时出错!", e);
            Thread.currentThread().interrupt();
        } finally {
            executorService.shutdown();
        }
    }

}
