/*
 * Copyright (c) 2019 huipei.x
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.xphsc.xpack.handler.impl;

import cn.xphsc.xpack.exception.ExceptionEnum;
import cn.xphsc.xpack.exception.XpackException;
import com.spotify.docker.client.DefaultDockerClient;
import com.spotify.docker.client.DockerClient;
import com.spotify.docker.client.LoggingPushHandler;
import com.spotify.docker.client.messages.ProgressMessage;
import com.spotify.docker.client.messages.RegistryAuth;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import cn.xphsc.xpack.domain.DockerGoalEnum;
import cn.xphsc.xpack.domain.PlatformEnum;
import cn.xphsc.xpack.domain.SkipErrorEnum;
import cn.xphsc.xpack.domain.Docker;
import cn.xphsc.xpack.domain.PackInfo;
import cn.xphsc.xpack.handler.AbstractPackHandler;
import cn.xphsc.xpack.utils.Logger;
import cn.xphsc.xpack.utils.Templates;
import org.apache.commons.lang3.ArrayUtils;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.io.RawInputStreamFacade;

/**
 * @author <a href="xiongpeih@163.com">huipei.x</a>
 * @description:
 * @since 1.0.0
 */
public class DockerPackHandler extends AbstractPackHandler {


    private static final String DOCKER_FILE = "Dockerfile";

    /**
     * Docker Client.
     */
    private DockerClient dockerClient;

    /**
     * Mirror name.
     */
    private String imageName;

    /**
     * The method of Docker construction and packaging according to the relevant parameters of packaging.
     *
     */
    @Override
    public void pack(PackInfo packInfo) {
        super.packInfo = packInfo;

        try {
            this.doPack();
        } catch (Exception e) {
            Logger.error("xpack 执行 Docker 构建失败！", e);
            throw e;
        } finally {
            this.clean();
        }
    }

    private void doPack() {
        if (SkipErrorEnum.TRUE == packInfo.getSkipError()) {
            try {
                this.doBuild();
                this.printFinished();
            } catch (Exception e) {
                Logger.error(e.getMessage());
            }
            return;
        }

        // If SkipError is false, it means that once an exception is encountered, an exception needs to be thrown without try-catch processing.
        if (SkipErrorEnum.FALSE == packInfo.getSkipError()) {
            this.doBuild();
            this.printFinished();
            return;
        }

        // Finally, if skipError is not configured, the default processing mode is followed, and if the docker environment is not detected, the build is skipped directly.
        try {
            this.checkDockerEnv();
        } catch (Exception e) {
            Logger.error(e.getMessage());
            return;
        }
        this.doBuildWithoutCheck();
        this.printFinished();
    }

    /**
     *
     Check whether the Docker environment meets the requirements of the build.
     */
    private void checkDockerEnv() {
        try {
            this.dockerClient = DefaultDockerClient.fromEnv().build();
            this.dockerClient.ping();
        } catch (Exception e) {
            throw new XpackException(ExceptionEnum.NO_DOCKER.getMsg(), e);
        }
    }

    /**
     * Initialize the Dockerfile file and copy the jar package.
     */
    private void initDockerfileAndJar() {
        super.createPlatformCommonDir(PlatformEnum.DOCKER);
        try {
            this.copyDockerfile();
        } catch (IOException e) {
            throw new XpackException(ExceptionEnum.NO_DOCKERFILE.getMsg(), e);
        }
    }

    /**
     * Build a Docker image of the service.
     */
    private void buildImage() {
        try {
            this.imageName = super.packInfo.getDocker().getImageName();
            Logger.info("正在构建【" + this.imageName + "】镜像...");
           //String imageId = dockerClient.build(Paths.get(super.platformPath), imageName, this::printProgress);
           String imageId = dockerClient.build(Paths.get(super.platformPath), imageName);
            Logger.info("构建【" + this.imageName + "】镜像完毕，镜像ID: " + imageId);
        } catch (Exception e) {
            throw new XpackException(ExceptionEnum.DOCKER_BUILD_EXCEPTION.getMsg(), e);
        }
    }

    /**
     *
     Export the image of the service as'.tar'package.
     */
    private void saveImage() {
        try {
            String imageTar =  super.packInfo.getDocker().getImageTarName() + ".tar";
            Logger.info("正在导出 Docker 镜像包: " + imageTar + " ...");
            try (InputStream imageInput = dockerClient.save(this.imageName)) {
                FileUtils.copyStreamToFile(new RawInputStreamFacade(imageInput),
                        new File(super.platformPath + File.separator + imageTar));
            }
            Logger.info("从 Docker 中导出镜像包 " + imageTar + " 成功.");
            this.handleFilesAndCompress();
        } catch (Exception  e) {
            throw new XpackException(ExceptionEnum.DOCKER_SAVE_EXCEPTION.getMsg(), e);
        }
    }

    /**
     * Relevant files that need to be packaged are compressed into tar.gz format.
     */
    private void handleFilesAndCompress() throws IOException {
        FileUtils.forceMkdir(new File(super.platformPath + File.separator + "docs"));
        super.copyFiles("docker/README.md", "README.md");
        super.compress(PlatformEnum.DOCKER);
    }

    /**
     *
     Label the mirror with a `registry'prefix to facilitate subsequent image push.
     */
    private String tagImage(String registry) {
        try {
            String imageTagName = registry + "/" + this.imageName;
            dockerClient.tag(this.imageName, imageTagName, true);
            Logger.info("已对本次构建的镜像打了标签，标签为：【" + imageTagName + "】.");
            return imageTagName;
        } catch (Exception e) {
            throw new XpackException(ExceptionEnum.DOCKER_TAG_EXCEPTION.getMsg(), e);
        }
    }

    /**
     * Push a Docker image to a remote warehouse.
     */
    private void pushImage() {
        this.initDockercfgFile();

        Logger.info("正在校验推送镜像时需要的 registry 授权是否合法...");
        try {
            RegistryAuth auth = RegistryAuth.builder().build();
            int statusCode = dockerClient.auth(auth);
             int code=200;
            if (statusCode != code) {
                Logger.warn("校验 registry 授权不通过，不能推送镜像到远程镜像仓库中.");
                return;
            }

            // Determine whether registry is configured or not. If it is not configured, it is assumed that it is pushed to dockerhub by default, and there is no need to label it.
            String registry = super.packInfo.getDocker().getRegistry();
            final String imageTagName = StringUtils.isBlank(registry) ? this.imageName : this.tagImage(registry);

            Logger.info("正在推送标签为【" + imageTagName + "】的镜像到远程仓库中...");

            dockerClient.push(imageTagName, new LoggingPushHandler(imageTagName), auth);
            Logger.info("推送标签为【" + imageTagName + "】的镜像到远程仓库中成功.");
        } catch (Exception e) {
            throw new XpackException(ExceptionEnum.DOCKER_PUSH_EXCEPTION.getMsg(), e);
        }
    }


    private void doBuild() {
        this.checkDockerEnv();
        this.doBuildWithoutCheck();
    }

    /**
     * Docker construction related judgment and processing, this method does not check the Docker environment.
     */
    private void doBuildWithoutCheck() {
        this.initDockerfileAndJar();
        this.buildImage();

        // If docker's configuration information is empty, it is directly considered to refer to the construction of the mirror.
        String[] goalTypes;
        Docker dockerInfo = super.packInfo.getDocker();
        if (dockerInfo == null || (goalTypes = dockerInfo.getExtraGoals()) == null || goalTypes.length == 0) {
            Logger.debug("在 xpack 中未配置  docker 额外构建目标类型'goalTypes'的值，只会构建镜像.");
            return;
        }

        Set<DockerGoalEnum> goalEnumSet = new HashSet<>(4);
        for (String goal : goalTypes) {
            DockerGoalEnum goalEnum = DockerGoalEnum.of(goal);
            if (goalEnum != null) {
                goalEnumSet.add(goalEnum);
            }
        }

        if (goalEnumSet.isEmpty()) {
            Logger.warn("在 xpack 中配置 docker 的额外构建目标类型'goalTypes'的值不是 save 或者 push，将忽略后续的构建.");
            return;
        }

        if (goalEnumSet.size() == 1) {
            if (goalEnumSet.contains(DockerGoalEnum.SAVE)) {
                this.saveImage();
            } else if (goalEnumSet.contains(DockerGoalEnum.PUSH)) {
                this.pushImage();
            }
        } else {
            this.saveImage();
            this.pushImage();
        }
    }


    private void copyDockerfile() throws IOException {
        Docker docker = super.packInfo.getDocker();
        if (StringUtils.isBlank(docker.getDockerfile())) {
            Logger.info("将使用 xpack 默认提供的 Dockerfile 文件来构建镜像.");
            Map<String, Object> context = super.buildBaseTemplateContextMap();
            context.put("jdkImage", docker.getFromImage());
            context.put("valume", this.buildVolumes(docker.getVolumes()));
            context.put("customCommands", this.buildCustomCommands(docker.getCustomCommands()));
            context.put("expose", this.buildExpose(docker.getExpose()));
            Templates.renderFile("docker/" + DOCKER_FILE, context,
                    super.platformPath + File.separator + DOCKER_FILE);
            return;
        }

        Logger.info("开始渲染你自定义的 Dockerfile 文件中的内容.");
        String dockerFilePath = docker.getDockerfile();
        File dockerFile = new File(super.isRootPath(dockerFilePath) ? DOCKER_FILE : dockerFilePath);
        if (!dockerFile.exists() || dockerFile.isDirectory()) {

            throw new XpackException(ExceptionEnum.NO_DOCKERFILE.getMsg());
        }

        FileUtils.copyFileToDirectory(dockerFile, new File(super.platformPath));
    }

    /**
     * Splice VOLUME strings based on volumes arrays.
     */
    private String buildVolumes(String[] volumes) {
        return ArrayUtils.isNotEmpty(volumes)
                ? "VOLUME [\"" + StringUtils.join(volumes, "\", \"") + "\"]\n" : "";
    }

    /**
     * Splice the string of EXPOSE according to the value of port expose to be exposed.
     */
    private String buildExpose(String expose) {
        return StringUtils.isEmpty(expose) ? "" : "\nEXPOSE " + expose.trim() + "\n";
    }

    /**
     *According to the custom command array customCommands, all kinds of command strings in Dockerfile are spliced together, and one command has only one line.
     *
     */
    private String buildCustomCommands(String[] customCommands) {
        StringBuilder sb = new StringBuilder();
        if (ArrayUtils.isNotEmpty(customCommands)) {
            for (String command : customCommands) {
                sb.append(command).append("\n");
            }
        }
        return sb.toString();
    }

    /**
     * Create a `dockercfg'file under the user directory of the current operating system. If not, initialize an empty file or leave it alone.
     */
    private void initDockercfgFile() {
        File dockercfgFile = new File(System.getProperty("user.home") + File.separator + ".dockercfg");
        if (!dockercfgFile.exists()) {
            try {
                org.apache.commons.io.FileUtils.touch(dockercfgFile);
            } catch (IOException e) {
                Logger.error("初始化 ~/.dockercfg 文件失败！", e);
            }
        }
    }

    /**
     Print Docker progress.
     */
   private void printProgress(ProgressMessage msg) {
         String progress = msg.progress();
        if (StringUtils.isNotBlank(progress)) {
            Logger.info(progress);
        }
    }


    private void printFinished() {
        Logger.debug("xpack 关于 Docker 的相关构建操作执行完毕.");
    }


    private void clean() {
        if (this.dockerClient != null) {
            this.dockerClient.close();
        }

        try {
            FileUtils.forceDelete(super.platformPath);
        } catch (Exception e) {
            Logger.debug("删除清除 docker 下的临时文件失败.");
        }
    }

}
