package cn.zcltd.http;

import cn.zcltd.http.response.HttpResponseHandler;
import cn.zcltd.http.response.StringHttpResponseResult;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * http(s)请求工具类
 */
public class HttpUtil {
    protected static final Logger log = LoggerFactory.getLogger(HttpUtil.class);
    protected static boolean printLog = false;
    protected static boolean printFile = false;
    protected static String reqFilePath = HttpUtil.class.getResource("/").getPath();

    public static String getQueryString(Map<String, String> queryParamMap) {
        StringBuilder querySb = new StringBuilder();
        for (Map.Entry<String, String> entry : queryParamMap.entrySet()) {
            querySb.append("&").append(entry.getKey().trim()).append("=").append(entry.getValue().trim());
        }
        return querySb.length() > 0 ? querySb.substring(1) : querySb.toString();
    }

    public static <T> T reqeust(HttpMethod httpMethod, String url, Map<String, String> queryParamMap, Map<String, String> headerMap, HttpEntity bodyEntity, HttpResponseHandler responseHandler) {
        try {
            HttpRequester httpRequester = HttpRequester.create();
            httpRequester.setHttpMethod(httpMethod).setUrl(url)
                    .addQueryParams(queryParamMap)
                    .addHeaders(headerMap)
                    .setBodyEntity(bodyEntity)
                    .setResponseHandler(responseHandler);
            httpRequester.getResponseHandler().setPrintLog(printLog);
            httpRequester.getResponseHandler().setPrintFile(printFile);
            httpRequester.getResponseHandler().setReqFilePath(reqFilePath);
            return (T) httpRequester.execute();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult get(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.GET, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        try {
            List<NameValuePair> list = new ArrayList<>();
            for (Map.Entry<String, Object> stringStringEntry : bodyParamMap.entrySet()) {
                list.add(new BasicNameValuePair(stringStringEntry.getKey(), stringStringEntry.getValue().toString()));
            }
            UrlEncodedFormEntity bodyEntity = new UrlEncodedFormEntity(list, "UTF-8");
            return (StringHttpResponseResult) reqeust(HttpMethod.POST, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        try {
            ByteArrayEntity bodyEntity = new ByteArrayEntity(bodyParamBytes);
            return (StringHttpResponseResult) reqeust(HttpMethod.POST, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap, Map<String, File> fileMap) {
        try {
            MultipartEntityBuilder multipartEntityBuilder = MultipartEntityBuilder.create();
            if (fileMap != null && fileMap.size() > 0) {
                for (Map.Entry<String, File> entry : fileMap.entrySet()) {
                    String paramName = entry.getKey();
                    File file = entry.getValue();
                    String fileName = file.getName();
                    fileName = URLEncoder.encode(fileName, "UTF-8");
                    ContentType contentType = ContentType.create("application/octet-stream", Charset.forName("UTF-8"));
                    multipartEntityBuilder.addBinaryBody(paramName, file, contentType, fileName);
                }
            }
            if (bodyParamMap != null && bodyParamMap.size() > 0) {
                for (Map.Entry<String, Object> entry : bodyParamMap.entrySet()) {
                    multipartEntityBuilder.addTextBody(entry.getKey(), entry.getValue().toString());
                }
            }
            HttpEntity bodyEntity = multipartEntityBuilder.build();

            return (StringHttpResponseResult) reqeust(HttpMethod.POST, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, String bodyParamName, byte[] bodyParamBytes, Map<String, File> fileMap) {
        try {
            MultipartEntityBuilder multipartEntityBuilder = MultipartEntityBuilder.create();
            if (fileMap != null && fileMap.size() > 0) {
                for (Map.Entry<String, File> entry : fileMap.entrySet()) {
                    String paramName = entry.getKey();
                    File file = entry.getValue();
                    String fileName = file.getName();
                    fileName = URLEncoder.encode(fileName, "UTF-8");
                    ContentType contentType = ContentType.create("application/octet-stream; charset=UTF-8");
                    multipartEntityBuilder.addBinaryBody(paramName, file, contentType, fileName);
                }
            }
            multipartEntityBuilder.addBinaryBody(bodyParamName, bodyParamBytes);
            HttpEntity bodyEntity = multipartEntityBuilder.build();

            return (StringHttpResponseResult) reqeust(HttpMethod.POST, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap, String fileName, File file) {
        try {
            Map<String, File> fileMap = new HashMap<>();
            fileMap.put(fileName, file);
            return post(url, queryParamMap, headerMap, bodyParamMap, fileMap);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, String bodyParamName, byte[] bodyParamBytes, String fileName, File file) {
        try {
            Map<String, File> fileMap = new HashMap<>();
            fileMap.put(fileName, file);
            return post(url, queryParamMap, headerMap, bodyParamName, bodyParamBytes, fileMap);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult put(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        try {
            List<NameValuePair> list = new ArrayList<>();
            for (Map.Entry<String, Object> stringStringEntry : bodyParamMap.entrySet()) {
                list.add(new BasicNameValuePair(stringStringEntry.getKey(), stringStringEntry.getValue().toString()));
            }
            UrlEncodedFormEntity bodyEntity = new UrlEncodedFormEntity(list);
            return (StringHttpResponseResult) reqeust(HttpMethod.PUT, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult put(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        try {
            ByteArrayEntity bodyEntity = new ByteArrayEntity(bodyParamBytes);
            return (StringHttpResponseResult) reqeust(HttpMethod.PUT, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult head(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.HEAD, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult delete(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.DELETE, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult patch(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        try {
            List<NameValuePair> list = new ArrayList<>();
            for (Map.Entry<String, Object> stringStringEntry : bodyParamMap.entrySet()) {
                list.add(new BasicNameValuePair(stringStringEntry.getKey(), stringStringEntry.getValue().toString()));
            }
            UrlEncodedFormEntity bodyEntity = new UrlEncodedFormEntity(list);
            return (StringHttpResponseResult) reqeust(HttpMethod.PATCH, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult patch(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        try {
            ByteArrayEntity bodyEntity = new ByteArrayEntity(bodyParamBytes);
            return (StringHttpResponseResult) reqeust(HttpMethod.PATCH, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult options(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.OPTIONS, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static StringHttpResponseResult trace(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.TRACE, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public static boolean isPrintLog() {
        return printLog;
    }

    public static void setPrintLog(boolean printLog) {
        HttpUtil.printLog = printLog;
    }

    public static boolean isPrintFile() {
        return printFile;
    }

    public static void setPrintFile(boolean printFile) {
        HttpUtil.printFile = printFile;
    }

    public static String getReqFilePath() {
        return reqFilePath;
    }

    public static void setReqFilePath(String reqFilePath) {
        HttpUtil.reqFilePath = reqFilePath;
    }

    public static SSLConnectionSocketFactory getSslcsf() {
        return HttpRequester.getSslcsf();
    }

    public static void setSslcsf(SSLConnectionSocketFactory sslcsf) {
        HttpRequester.setSslcsf(sslcsf);
    }

    public static RequestConfig getRequestConfig() {
        return HttpRequester.getRequestConfig();
    }

    public static void setRequestConfig(RequestConfig requestConfig) {
        HttpRequester.setRequestConfig(requestConfig);
    }

    public static PoolingHttpClientConnectionManager getCm() {
        return HttpRequester.getCm();
    }

    public static void setCm(PoolingHttpClientConnectionManager cm) {
        HttpRequester.setCm(cm);
    }
}