package cn.zcltd.http;

import cn.zcltd.http.response.HttpResponseHandler;
import cn.zcltd.http.response.HttpResponseResult;
import cn.zcltd.http.response.StringHttpResponseHandler;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.*;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.ssl.SSLContextBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * http(s)请求工具类
 */
public class HttpRequester {
    protected static final Logger log = LoggerFactory.getLogger(HttpRequester.class);
    private HttpResponseHandler responseHandlerDefault = new StringHttpResponseHandler("UTF-8");
    private static HttpRequester instance = new HttpRequester();

    private static PoolingHttpClientConnectionManager cm;
    private static SSLConnectionSocketFactory sslcsf;
    private static RequestConfig requestConfig;

    private HttpClient httpClient;

    private HttpMethod httpMethod;
    private String url;
    private Map<String, String> queryParamMap = new LinkedHashMap<>();
    private Map<String, String> headerMap = new HashMap<>();
    private HttpEntity bodyEntity;

    private HttpResponseHandler responseHandler = responseHandlerDefault;

    static {
        try {
            SSLContextBuilder builder = new SSLContextBuilder();

            // 全部信任 不做身份鉴定
            builder.loadTrustMaterial(null, new TrustStrategy() {
                @Override
                public boolean isTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
                    return true;
                }
            });

            sslcsf = new SSLConnectionSocketFactory(builder.build(), new String[]{"SSLv2Hello", "SSLv3", "TLSv1", "TLSv1.2"}, null, NoopHostnameVerifier.INSTANCE);
            Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                    .register("http", new PlainConnectionSocketFactory())
                    .register("https", sslcsf)
                    .build();
            cm = new PoolingHttpClientConnectionManager(registry);
            cm.setMaxTotal(200);//max connection
            requestConfig = RequestConfig.DEFAULT;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    private HttpRequester() {
        httpClient = HttpClientBuilder.create()
                .setSSLHostnameVerifier(new HostnameVerifier() {
                    @Override
                    public boolean verify(String s, SSLSession sslSession) {
                        return true;
                    }
                })
                .setSSLSocketFactory(sslcsf)
                .setConnectionManager(cm)
                .setConnectionManagerShared(true)
                .build();
    }

    public static HttpRequester create() {
        instance.url = null;
        instance.httpMethod = null;
        instance.queryParamMap.clear();
        instance.headerMap.clear();
        instance.bodyEntity = null;
        instance.responseHandler = instance.responseHandlerDefault;
        return instance;
    }

    public HttpRequester setHttpMethod(HttpMethod httpMethod) {
        if (httpMethod == null) {
            throw new RuntimeException("httpMethod must not be null");
        }
        this.httpMethod = httpMethod;
        return this;
    }

    public HttpRequester setUrl(String url) {
        if (url == null || url.length() == 0) {
            throw new RuntimeException("url must not be null");
        }
        this.url = url;
        return this;
    }

    public HttpRequester addQueryParam(String key, String value) {
        if (key != null && key.length() > 0) {
            this.queryParamMap.put(key, value);
        }
        return this;
    }

    public HttpRequester addQueryParams(Map<String, String> queryParams) {
        if (queryParams != null) {
            this.queryParamMap.putAll(queryParams);
        }
        return this;
    }

    public HttpRequester addHeader(String key, String value) {
        if (key != null && key.length() > 0) {
            this.headerMap.put(key, value);
        }
        return this;
    }

    public HttpRequester addHeaders(Map<String, String> headers) {
        if (headers != null) {
            this.headerMap.putAll(headers);
        }
        return this;
    }

    public HttpRequester setBodyEntity(HttpEntity bodyEntity) {
        if (bodyEntity != null) {
            this.bodyEntity = bodyEntity;
        }
        return this;
    }

    public HttpResponseHandler getResponseHandler() {
        return responseHandler;
    }

    public HttpRequester setResponseHandler(HttpResponseHandler responseHandler) {
        if (responseHandler != null) {
            this.responseHandler = responseHandler;
        }
        return this;
    }

    public HttpResponseResult execute() throws Exception {
        HttpRequestBase request = getHttpUriRequest();
        request.setConfig(requestConfig);
        for (Map.Entry<String, String> header : headerMap.entrySet()) {
            request.addHeader(header.getKey(), header.getValue());
        }
        HttpResponse response = httpClient.execute(request);
        return responseHandler.handler(request, response);
    }

    private HttpRequestBase getHttpUriRequest() {
        String url = this.url;
        String queryStr = HttpUtil.getQueryString(this.queryParamMap);
        String queryStrChar = url.indexOf("?") > 0 ? "&" : "?";
        url = queryStr.length() > 0 ? url + queryStrChar + queryStr : url;

        switch (this.httpMethod) {
            case GET:
                return new HttpGet(url);
            case POST:
                HttpPost httpPost = new HttpPost(url);
                httpPost.setEntity(bodyEntity);
                return httpPost;
            case PUT:
                HttpPut httpPut = new HttpPut(url);
                httpPut.setEntity(bodyEntity);
                return httpPut;
            case HEAD:
                return new HttpHead(url);
            case DELETE:
                return new HttpDelete(url);
            case PATCH:
                HttpPatch httpPatch = new HttpPatch(url);
                httpPatch.setEntity(bodyEntity);
                return httpPatch;
            case OPTIONS:
                return new HttpOptions(url);
            case TRACE:
                return new HttpTrace(url);
            default:
                return new HttpGet(url);
        }
    }

    public static SSLConnectionSocketFactory getSslcsf() {
        return sslcsf;
    }

    public static void setSslcsf(SSLConnectionSocketFactory sslcsf) {
        HttpRequester.sslcsf = sslcsf;
    }

    public static RequestConfig getRequestConfig() {
        return requestConfig;
    }

    public static void setRequestConfig(RequestConfig requestConfig) {
        HttpRequester.requestConfig = requestConfig;
    }

    public static PoolingHttpClientConnectionManager getCm() {
        return cm;
    }

    public static void setCm(PoolingHttpClientConnectionManager cm) {
        HttpRequester.cm = cm;
    }
}