package cn.zcltd.http;

import cn.zcltd.btg.sutil.EmptyUtil;
import cn.zcltd.btg.sutil.FileUtil;
import cn.zcltd.btg.sutil.PathUtil;
import cn.zcltd.http.response.HttpResponseHandler;
import cn.zcltd.http.response.HttpResponseResult;
import cn.zcltd.http.response.StringHttpResponseHandler;
import cn.zcltd.http.response.StringHttpResponseResult;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.*;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.ssl.SSLContextBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;
import java.io.File;
import java.io.FileOutputStream;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * http(s)请求工具类
 */
public class HttpRequester {
    private static final Logger log = LoggerFactory.getLogger(HttpRequester.class);
    private static final HttpResponseHandler defaultHttpResponseHandler = new StringHttpResponseHandler();

    private PoolingHttpClientConnectionManager cm;
    private SSLConnectionSocketFactory sslcsf;
    private RequestConfig requestConfig;

    private HttpClient httpClient;
    private HttpMethod httpMethod;
    private String url;
    private Map<String, String> queryParamMap;
    private Map<String, String> headerMap;
    private HttpEntity bodyEntity;

    private HttpResponseHandler responseHandler = defaultHttpResponseHandler;

    private Boolean printLog = false;
    private Boolean printFile = false;
    private String reqFilePath = PathUtil.getRootClassPath();

    /**
     * 构建基础请求环境
     */
    public HttpRequester() {
        try {
            SSLContextBuilder builder = new SSLContextBuilder();
            builder.loadTrustMaterial(null, new TrustStrategy() {// 全部信任 不做身份鉴定
                @Override
                public boolean isTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
                    return true;
                }
            });
            sslcsf = new SSLConnectionSocketFactory(builder.build(), new String[]{"SSLv2Hello", "SSLv3", "TLSv1", "TLSv1.2"}, null, NoopHostnameVerifier.INSTANCE);

            Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                    .register("http", new PlainConnectionSocketFactory())
                    .register("https", sslcsf)
                    .build();
            cm = new PoolingHttpClientConnectionManager(registry);
            cm.setMaxTotal(200);

            requestConfig = RequestConfig.DEFAULT;

            httpClient = HttpClientBuilder.create()
                    .setSSLHostnameVerifier(new HostnameVerifier() {
                        @Override
                        public boolean verify(String s, SSLSession sslSession) {
                            return true;
                        }
                    })
                    .setSSLSocketFactory(sslcsf)
                    .setConnectionManager(cm)
                    .setConnectionManagerShared(true)
                    .build();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 添加url参数
     *
     * @param key   名称
     * @param value 值
     * @return 自身
     */
    public HttpRequester addQueryParam(String key, String value) {
        if (key != null && key.length() > 0) {
            this.queryParamMap.put(key, value);
        }
        return this;
    }

    /**
     * 添加多个url参数
     *
     * @param queryParams 参数map
     * @return 自身
     */
    public HttpRequester addQueryParams(Map<String, String> queryParams) {
        if (queryParams != null) {
            this.queryParamMap.putAll(queryParams);
        }
        return this;
    }

    /**
     * 添加header
     *
     * @param key   名称
     * @param value 值
     * @return 自身
     */
    public HttpRequester addHeader(String key, String value) {
        if (key != null && key.length() > 0) {
            this.headerMap.put(key, value);
        }
        return this;
    }

    /**
     * 添加多个header
     *
     * @param headers header参数map
     * @return 自身
     */
    public HttpRequester addHeaders(Map<String, String> headers) {
        if (headers != null) {
            this.headerMap.putAll(headers);
        }
        return this;
    }

    /**
     * 执行请求
     *
     * @return 请求结果
     * @throws Exception 异常
     */
    public HttpResponseResult execute() throws Exception {
        HttpRequestBase request = getHttpUriRequest();
        request.setConfig(requestConfig);
        for (Map.Entry<String, String> header : headerMap.entrySet()) {
            request.addHeader(header.getKey(), header.getValue());
        }
        HttpResponse response = httpClient.execute(request);
        HttpResponseResult httpResponseResult = responseHandler.handler(request, response);
        print(request, response);
        return httpResponseResult;
    }

    /**
     * 重置
     */
    public HttpRequester reset() {
        httpMethod = HttpMethod.GET;
        url = null;
        queryParamMap = new LinkedHashMap<>();
        headerMap = new HashMap<>();
        bodyEntity = null;
        responseHandler = defaultHttpResponseHandler;
        return this;
    }

    /**
     * 构建request对象
     *
     * @return request对象
     */
    private HttpRequestBase getHttpUriRequest() {
        String url = this.url;
        String queryStr = getQueryString(this.queryParamMap);
        String queryStrChar = url.indexOf("?") > 0 ? "&" : "?";
        url = queryStr.length() > 0 ? url + queryStrChar + queryStr : url;

        switch (this.httpMethod) {
            case GET:
                return new HttpGet(url);
            case POST:
                HttpPost httpPost = new HttpPost(url);
                httpPost.setEntity(bodyEntity);
                return httpPost;
            case PUT:
                HttpPut httpPut = new HttpPut(url);
                httpPut.setEntity(bodyEntity);
                return httpPut;
            case HEAD:
                return new HttpHead(url);
            case DELETE:
                return new HttpDelete(url);
            case PATCH:
                HttpPatch httpPatch = new HttpPatch(url);
                httpPatch.setEntity(bodyEntity);
                return httpPatch;
            case OPTIONS:
                return new HttpOptions(url);
            case TRACE:
                return new HttpTrace(url);
            default:
                return new HttpGet(url);
        }
    }

    /**
     * 发送请求
     *
     * @param httpMethod      请求方法
     * @param url             请求地址
     * @param queryParamMap   url参数
     * @param headerMap       header参数
     * @param bodyEntity      body内容
     * @param responseHandler 结果解析器
     * @param <T>             返回结果类
     * @return 返回结果
     */
    public <T> T reqeust(HttpMethod httpMethod, String url, Map<String, String> queryParamMap, Map<String, String> headerMap, HttpEntity bodyEntity, HttpResponseHandler responseHandler) {
        try {
            this.reset()
                    .setHttpMethod(httpMethod)
                    .setUrl(url)
                    .addQueryParams(queryParamMap)
                    .addHeaders(headerMap)
                    .setBodyEntity(bodyEntity)
                    .setResponseHandler(responseHandler);
            return (T) this.execute();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 发送get请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @return 返回结果
     */
    public StringHttpResponseResult get(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.GET, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 发送post请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @param bodyParamMap  body内容
     * @return 返回结果
     */
    public StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        try {
            List<NameValuePair> list = new ArrayList<>();
            for (Map.Entry<String, Object> stringStringEntry : bodyParamMap.entrySet()) {
                list.add(new BasicNameValuePair(stringStringEntry.getKey(), stringStringEntry.getValue().toString()));
            }
            UrlEncodedFormEntity bodyEntity = new UrlEncodedFormEntity(list, "UTF-8");
            return (StringHttpResponseResult) reqeust(HttpMethod.POST, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 发送post请求
     *
     * @param url            请求地址
     * @param queryParamMap  url参数
     * @param headerMap      header参数
     * @param bodyParamBytes body内容
     * @return 返回结果
     */
    public StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        try {
            ByteArrayEntity bodyEntity = new ByteArrayEntity(bodyParamBytes);
            return (StringHttpResponseResult) reqeust(HttpMethod.POST, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送post请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @param bodyParamMap  body内容
     * @param fileMap       文件map
     * @return 返回结果
     */
    public StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap, Map<String, File> fileMap) {
        try {
            MultipartEntityBuilder multipartEntityBuilder = MultipartEntityBuilder.create();
            if (fileMap != null && fileMap.size() > 0) {
                for (Map.Entry<String, File> entry : fileMap.entrySet()) {
                    String paramName = entry.getKey();
                    File file = entry.getValue();
                    String fileName = file.getName();
                    fileName = URLEncoder.encode(fileName, "UTF-8");
                    ContentType contentType = ContentType.create("application/octet-stream", Charset.forName("UTF-8"));
                    multipartEntityBuilder.addBinaryBody(paramName, file, contentType, fileName);
                }
            }
            if (bodyParamMap != null && bodyParamMap.size() > 0) {
                for (Map.Entry<String, Object> entry : bodyParamMap.entrySet()) {
                    multipartEntityBuilder.addTextBody(entry.getKey(), entry.getValue().toString());
                }
            }
            HttpEntity bodyEntity = multipartEntityBuilder.build();

            return (StringHttpResponseResult) reqeust(HttpMethod.POST, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送post请求
     *
     * @param url            请求地址
     * @param queryParamMap  url参数
     * @param headerMap      header参数
     * @param bodyParamName  body参数名称
     * @param bodyParamBytes body内容
     * @param fileMap        文件map
     * @return 返回结果
     */
    public StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, String bodyParamName, byte[] bodyParamBytes, Map<String, File> fileMap) {
        try {
            MultipartEntityBuilder multipartEntityBuilder = MultipartEntityBuilder.create();
            if (fileMap != null && fileMap.size() > 0) {
                for (Map.Entry<String, File> entry : fileMap.entrySet()) {
                    String paramName = entry.getKey();
                    File file = entry.getValue();
                    String fileName = file.getName();
                    fileName = URLEncoder.encode(fileName, "UTF-8");
                    ContentType contentType = ContentType.create("application/octet-stream", Charset.forName("UTF-8"));
                    multipartEntityBuilder.addBinaryBody(paramName, file, contentType, fileName);
                }
            }
            multipartEntityBuilder.addBinaryBody(bodyParamName, bodyParamBytes);
            HttpEntity bodyEntity = multipartEntityBuilder.build();

            return (StringHttpResponseResult) reqeust(HttpMethod.POST, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送post请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @param bodyParamMap  body内容
     * @param fileName      文件名
     * @param file          文件
     * @return 返回结果
     */
    public StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap, String fileName, File file) {
        try {
            Map<String, File> fileMap = new HashMap<>();
            fileMap.put(fileName, file);
            return post(url, queryParamMap, headerMap, bodyParamMap, fileMap);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送post请求
     *
     * @param url            请求地址
     * @param queryParamMap  url参数
     * @param headerMap      header参数
     * @param bodyParamName  body参数名称
     * @param bodyParamBytes body内容
     * @param fileName       文件名
     * @param file           文件
     * @return 返回结果
     */
    public StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, String bodyParamName, byte[] bodyParamBytes, String fileName, File file) {
        try {
            Map<String, File> fileMap = new HashMap<>();
            fileMap.put(fileName, file);
            return post(url, queryParamMap, headerMap, bodyParamName, bodyParamBytes, fileMap);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送put请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @param bodyParamMap  body内容
     * @return 返回结果
     */
    public StringHttpResponseResult put(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        try {
            List<NameValuePair> list = new ArrayList<>();
            for (Map.Entry<String, Object> stringStringEntry : bodyParamMap.entrySet()) {
                list.add(new BasicNameValuePair(stringStringEntry.getKey(), stringStringEntry.getValue().toString()));
            }
            UrlEncodedFormEntity bodyEntity = new UrlEncodedFormEntity(list);
            return (StringHttpResponseResult) reqeust(HttpMethod.PUT, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送put请求
     *
     * @param url            请求地址
     * @param queryParamMap  url参数
     * @param headerMap      header参数
     * @param bodyParamBytes body内容
     * @return 返回结果
     */
    public StringHttpResponseResult put(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        try {
            ByteArrayEntity bodyEntity = new ByteArrayEntity(bodyParamBytes);
            return (StringHttpResponseResult) reqeust(HttpMethod.PUT, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送head请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @return 返回结果
     */
    public StringHttpResponseResult head(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.HEAD, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送delete请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @return 返回结果
     */
    public StringHttpResponseResult delete(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.DELETE, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送patch请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @param bodyParamMap  body内容
     * @return 返回结果
     */
    public StringHttpResponseResult patch(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        try {
            List<NameValuePair> list = new ArrayList<>();
            for (Map.Entry<String, Object> stringStringEntry : bodyParamMap.entrySet()) {
                list.add(new BasicNameValuePair(stringStringEntry.getKey(), stringStringEntry.getValue().toString()));
            }
            UrlEncodedFormEntity bodyEntity = new UrlEncodedFormEntity(list);
            return (StringHttpResponseResult) reqeust(HttpMethod.PATCH, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送patch请求
     *
     * @param url            请求地址
     * @param queryParamMap  url参数
     * @param headerMap      header参数
     * @param bodyParamBytes body内容
     * @return 返回结果
     */
    public StringHttpResponseResult patch(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        try {
            ByteArrayEntity bodyEntity = new ByteArrayEntity(bodyParamBytes);
            return (StringHttpResponseResult) reqeust(HttpMethod.PATCH, url, queryParamMap, headerMap, bodyEntity, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送options请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @return 返回结果
     */
    public StringHttpResponseResult options(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.OPTIONS, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 发送trace请求
     *
     * @param url           请求地址
     * @param queryParamMap url参数
     * @param headerMap     header参数
     * @return 返回结果
     */
    public StringHttpResponseResult trace(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        try {
            return (StringHttpResponseResult) reqeust(HttpMethod.TRACE, url, queryParamMap, headerMap, null, null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 打印请求日志
     *
     * @param request  请求对象
     * @param response 响应对象
     */
    public void print(HttpUriRequest request, HttpResponse response) {
        try {
            StringBuilder resultSb = new StringBuilder("\n");

            resultSb.append(String.format("URL:%s\n", request.getURI()));

            resultSb.append(String.format("Method:%s\n", request.getMethod()));

            Header[] allHeaders = request.getAllHeaders();
            resultSb.append("RequestHeaders:\n");
            for (Header allHeader : allHeaders) {
                resultSb.append(String.format("\t%s:%s\n", allHeader.getName(), allHeader.getValue()));
            }

            resultSb.append("requestEntity:\n");
            if (EmptyUtil.isNotEmpty(bodyEntity)) {
                resultSb.append("\t").append(FileUtil.readToString(bodyEntity.getContent())).append("\n");
            }

            resultSb.append(String.format("ResponseCode:%s\n", response.getStatusLine().getStatusCode()));

            resultSb.append("ResponseHeaders:\n");
            for (Header allHeader : response.getAllHeaders()) {
                resultSb.append(String.format("\t%s:%s\n", allHeader.getName(), allHeader.getValue()));
            }

            resultSb.append("ResponseEntity:\n");
            resultSb.append(responseHandler.buildPrintBody());

            if (printLog) {
                log.debug(resultSb.toString());
            }

            if (printFile) {
                File dir = new File(reqFilePath);
                if (!dir.exists() || !dir.isDirectory()) {
                    boolean isSuccess = dir.mkdirs();
                    if (!isSuccess) {
                        throw new RuntimeException("directory does not exist and create the failure," + dir.getCanonicalPath());
                    }
                }

                String fileName = new SimpleDateFormat("yyyyMMddHHmmssSSS").format(new Date()) + request.getMethod() + "-" + URLEncoder.encode(request.getURI().getPath(), "UTF-8") + ".req";
                File file = new File(reqFilePath + fileName);
                FileOutputStream outputStream = new FileOutputStream(file);
                outputStream.write(resultSb.toString().getBytes());
                outputStream.close();
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    /**
     * 构建url参数
     *
     * @param queryParamMap url参数map
     * @return url的键值对字符串
     */
    public static String getQueryString(Map<String, String> queryParamMap) {
        StringBuilder querySb = new StringBuilder();
        for (Map.Entry<String, String> entry : queryParamMap.entrySet()) {
            querySb.append("&").append(entry.getKey().trim()).append("=").append(entry.getValue().trim());
        }
        return querySb.length() > 0 ? querySb.substring(1) : querySb.toString();
    }

    public PoolingHttpClientConnectionManager getCm() {
        return cm;
    }

    public HttpRequester setCm(PoolingHttpClientConnectionManager cm) {
        this.cm = cm;
        return this;
    }

    public SSLConnectionSocketFactory getSslcsf() {
        return sslcsf;
    }

    public HttpRequester setSslcsf(SSLConnectionSocketFactory sslcsf) {
        this.sslcsf = sslcsf;
        return this;
    }

    public RequestConfig getRequestConfig() {
        return requestConfig;
    }

    public HttpRequester setRequestConfig(RequestConfig requestConfig) {
        this.requestConfig = requestConfig;
        return this;
    }

    public HttpClient getHttpClient() {
        return httpClient;
    }

    public HttpRequester setHttpClient(HttpClient httpClient) {
        this.httpClient = httpClient;
        return this;
    }

    public HttpMethod getHttpMethod() {
        return httpMethod;
    }

    public HttpRequester setHttpMethod(HttpMethod httpMethod) {
        this.httpMethod = httpMethod;
        return this;
    }

    public String getUrl() {
        return url;
    }

    public HttpRequester setUrl(String url) {
        this.url = url;
        return this;
    }

    public Map<String, String> getQueryParamMap() {
        return queryParamMap;
    }

    public HttpRequester setQueryParamMap(Map<String, String> queryParamMap) {
        this.queryParamMap = queryParamMap;
        return this;
    }

    public Map<String, String> getHeaderMap() {
        return headerMap;
    }

    public HttpRequester setHeaderMap(Map<String, String> headerMap) {
        this.headerMap = headerMap;
        return this;
    }

    public HttpEntity getBodyEntity() {
        return bodyEntity;
    }

    public HttpRequester setBodyEntity(HttpEntity bodyEntity) {
        this.bodyEntity = bodyEntity;
        return this;
    }

    public HttpResponseHandler getResponseHandler() {
        return responseHandler;
    }

    public HttpRequester setResponseHandler(HttpResponseHandler responseHandler) {
        if (EmptyUtil.isNotEmpty(responseHandler)) {
            this.responseHandler = responseHandler;
        }
        return this;
    }

    public Boolean getPrintLog() {
        return printLog;
    }

    public HttpRequester setPrintLog(Boolean printLog) {
        this.printLog = printLog;
        return this;
    }

    public Boolean getPrintFile() {
        return printFile;
    }

    public HttpRequester setPrintFile(Boolean printFile) {
        this.printFile = printFile;
        return this;
    }

    public String getReqFilePath() {
        return reqFilePath;
    }

    public HttpRequester setReqFilePath(String reqFilePath) {
        this.reqFilePath = reqFilePath;
        return this;
    }
}