package cn.zcltd.http;

import cn.zcltd.http.response.HttpResponseHandler;
import cn.zcltd.http.response.StringHttpResponseResult;
import cn.zcltd.http.task.TaskBTGHttpRequesterTimeoutClean;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * http(s)请求工具类
 */
public class HttpUtil {
    private static final Logger log = LoggerFactory.getLogger(HttpUtil.class);

    private static final HttpRequester defaultRequester = new HttpRequester();
    private static final Map<String, HttpRequester> requesterMap = new ConcurrentHashMap<>();
    private static int maxClearTimeoutSeconds = 60;//过期requester清理机制触发周期(秒)

    static {
        TaskBTGHttpRequesterTimeoutClean.getInstance().start();
    }

    private HttpUtil() {

    }

    /**
     * 切换当前适用requester
     *
     * @param name 名称
     * @return 自身
     */
    public static HttpRequester use(String name) {
        HttpRequester requester = requesterMap.get(name);
        if (requester == null) {
            synchronized (HttpUtil.class) {
                if (requester == null) {
                    requester = new HttpRequester();
                    requesterMap.put(name, requester);
                }
            }
        }
        return requester;
    }

    /**
     * 切换当前适用requester
     *
     * @param name 名称
     * @return 自身
     */
    public static HttpRequester use(String name, RequestConfig requestConfig) {
        HttpRequester requester = requesterMap.get(name);
        if (requester == null) {
            synchronized (HttpUtil.class) {
                if (requester == null) {
                    requester = new HttpRequester();
                    requester.setRequestConfig(requestConfig);
                    requesterMap.put(name, requester);
                }
            }
        }
        return requester;
    }

    /**
     * 移除指定名称的requester
     *
     * @param name 名称
     * @return 被销毁的容器
     */
    public static HttpRequester remove(String name) {
        HttpRequester requester = requesterMap.remove(name);
        log.debug("remove http requester," + name);
        return requester;
    }

    /**
     * 清理过期的requester
     */
    public static void clearTimeout() {
        Iterator<Map.Entry<String, HttpRequester>> iterator = requesterMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, HttpRequester> entry = iterator.next();
            String name = entry.getKey();
            HttpRequester requester = entry.getValue();
            if (requester.isLost()) {
                iterator.remove();
                log.debug("remove timeout http requester," + name);
            }
        }
    }

    public static String getQueryString(Map<String, String> queryParamMap) {
        return HttpRequester.getQueryString(queryParamMap);
    }

    public static <T> T reqeust(HttpMethod httpMethod, String url, Map<String, String> queryParamMap, Map<String, String> headerMap, HttpEntity bodyEntity, HttpResponseHandler responseHandler) {
        return defaultRequester.reqeust(httpMethod, url, queryParamMap, headerMap, bodyEntity, responseHandler);
    }

    public static StringHttpResponseResult get(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        return defaultRequester.get(url, queryParamMap, headerMap);
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        return defaultRequester.post(url, queryParamMap, headerMap, bodyParamMap);
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        return defaultRequester.post(url, queryParamMap, headerMap, bodyParamBytes);
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap, Map<String, File> fileMap) {
        return defaultRequester.post(url, queryParamMap, headerMap, bodyParamMap, fileMap);
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, String bodyParamName, byte[] bodyParamBytes, Map<String, File> fileMap) {
        return defaultRequester.post(url, queryParamMap, headerMap, bodyParamName, bodyParamBytes, fileMap);
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap, String fileName, File file) {
        return defaultRequester.post(url, queryParamMap, headerMap, bodyParamMap, fileName, file);
    }

    public static StringHttpResponseResult post(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, String bodyParamName, byte[] bodyParamBytes, String fileName, File file) {
        return defaultRequester.post(url, queryParamMap, headerMap, bodyParamName, bodyParamBytes, fileName, file);
    }

    public static StringHttpResponseResult put(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        return defaultRequester.put(url, queryParamMap, headerMap, bodyParamMap);
    }

    public static StringHttpResponseResult put(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        return defaultRequester.put(url, queryParamMap, headerMap, bodyParamBytes);
    }

    public static StringHttpResponseResult head(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        return defaultRequester.head(url, queryParamMap, headerMap);
    }

    public static StringHttpResponseResult delete(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        return defaultRequester.delete(url, queryParamMap, headerMap);
    }

    public static StringHttpResponseResult patch(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, Map<String, Object> bodyParamMap) {
        return defaultRequester.patch(url, queryParamMap, headerMap, bodyParamMap);
    }

    public static StringHttpResponseResult patch(String url, Map<String, String> queryParamMap, Map<String, String> headerMap, byte[] bodyParamBytes) {
        return defaultRequester.patch(url, queryParamMap, headerMap, bodyParamBytes);
    }

    public static StringHttpResponseResult options(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        return defaultRequester.options(url, queryParamMap, headerMap);
    }

    public static StringHttpResponseResult trace(String url, Map<String, String> queryParamMap, Map<String, String> headerMap) {
        return defaultRequester.trace(url, queryParamMap, headerMap);
    }

    public static HttpRequester getDefaultRequester() {
        return defaultRequester;
    }

    public static Map<String, HttpRequester> getRequesterMap() {
        return requesterMap;
    }

    public static Boolean getPrintLog() {
        return defaultRequester.getPrintLog();
    }

    public static void setPrintLog(Boolean printLog) {
        defaultRequester.setPrintLog(printLog);
    }

    public static Boolean getPrintFile() {
        return defaultRequester.getPrintFile();
    }

    public static void setPrintFile(Boolean printFile) {
        defaultRequester.setPrintFile(printFile);
    }

    public static String getReqFilePath() {
        return defaultRequester.getReqFilePath();
    }

    public static void setReqFilePath(String reqFilePath) {
        defaultRequester.setReqFilePath(reqFilePath);
    }

    public long getActiveTimeMillis() {
        return defaultRequester.getActiveTimeMillis();
    }

    public void setActiveTimeMillis(long activeTimeMillis) {
        defaultRequester.setActiveTimeMillis(activeTimeMillis);
    }

    public static int getMaxClearTimeoutSeconds() {
        return maxClearTimeoutSeconds;
    }

    public static void setMaxClearTimeoutSeconds(int maxClearTimeoutSeconds) {
        HttpUtil.maxClearTimeoutSeconds = maxClearTimeoutSeconds;
    }
}