package cn.zcltd.btg.httpsession.impl;

import cn.zcltd.btg.core.exception.BtgRuntimeException;
import cn.zcltd.btg.httpsession.BTGSession;
import cn.zcltd.btg.httpsession.BTGSessionDao;
import cn.zcltd.btg.sutil.EmptyUtil;
import cn.zcltd.btg.sutil.FileUtil;
import cn.zcltd.btg.sutil.SerializableUtil;
import com.jfinal.plugin.activerecord.DbPro;
import com.jfinal.plugin.activerecord.Record;
import com.jfinal.plugin.activerecord.dialect.Dialect;
import com.jfinal.plugin.activerecord.dialect.OracleDialect;

import java.io.FileInputStream;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.util.Hashtable;
import java.util.List;

/**
 * 基于db的session存储实现
 */
public class BTGDBSessionDao implements BTGSessionDao {
    protected BTGDBSessionDaoConfig config;
    private boolean isInit = false;

    public BTGDBSessionDao() {
        this.config = new BTGDBSessionDaoConfig();
    }

    public BTGDBSessionDao(String arpName) {
        this.config = new BTGDBSessionDaoConfig(arpName);
    }

    public BTGDBSessionDao(BTGDBSessionDaoConfig sessionDBConfig) {
        this.config = sessionDBConfig;
    }

    @Override
    public void saveSession(BTGSession session) {
        initDb();

        if (EmptyUtil.isEmpty(session)) {
            throw new BtgRuntimeException("session is null");
        }

        this.config.getDbPro().deleteById(this.config.getTableName(), this.config.getSessionIdColumnName(), session.getId());

        Record sessionRecord = new Record();
        sessionRecord.set(this.config.getSessionIdColumnName(), session.getId());
        sessionRecord.set(this.config.getSessionObjColumnName(), SerializableUtil.obj2bytes(session));
        sessionRecord.set(this.config.getSessionLastActiveTimeColumnName(), session.getLastAccessedTime());
        sessionRecord.set(this.config.getSessionMaxInactiveIntervalColumnName(), session.getMaxInactiveInterval() * 1000);
        this.config.getDbPro().save(this.config.getTableName(), this.config.getSessionIdColumnName(), sessionRecord);
    }

    @Override
    public void deleteSession(String sessionId) {
        initDb();

        if (EmptyUtil.isEmpty(sessionId)) {
            throw new BtgRuntimeException("session id is null");
        }
        this.config.getDbPro().deleteById(this.config.getTableName(), this.config.getSessionIdColumnName(), sessionId);
    }

    @Override
    public BTGSession getSession(String sessionId) {
        initDb();

        if (EmptyUtil.isEmpty(sessionId)) {
            throw new BtgRuntimeException("session id is null");
        }

        Record sessionRecord = this.config.getDbPro().findById(this.config.getTableName(), this.config.getSessionIdColumnName(), sessionId);
        if (EmptyUtil.isEmpty(sessionRecord)) {
            return null;
        }

        byte[] sessionBytes = sessionRecord.getBytes(this.config.getSessionObjColumnName());
        BTGSession session = SerializableUtil.bytes2obj(sessionBytes);
        session.setLastAccessedTime(sessionRecord.getLong(this.config.getSessionLastActiveTimeColumnName()));

        return session;
    }

    @Override
    public void refreshSession(BTGSession session) {
        initDb();

        if (EmptyUtil.isEmpty(session)) {
            throw new BtgRuntimeException("session is null");
        }
        Record sessionRecord = new Record();
        sessionRecord.set(this.config.getSessionIdColumnName(), session.getId());
        sessionRecord.set(this.config.getSessionObjColumnName(), SerializableUtil.obj2bytes(session));
        sessionRecord.set(this.config.getSessionLastActiveTimeColumnName(), session.getLastAccessedTime());
        sessionRecord.set(this.config.getSessionMaxInactiveIntervalColumnName(), session.getMaxInactiveInterval() * 1000);
        this.config.getDbPro().update(this.config.getTableName(), this.config.getSessionIdColumnName(), sessionRecord);
    }

    @Override
    public void active(BTGSession session) {
        initDb();

        if (EmptyUtil.isEmpty(session)) {
            throw new BtgRuntimeException("session is null");
        }
        Record sessionRecord = new Record();
        sessionRecord.set(this.config.getSessionIdColumnName(), session.getId());
        sessionRecord.set(this.config.getSessionLastActiveTimeColumnName(), System.currentTimeMillis());
        sessionRecord.set(this.config.getSessionMaxInactiveIntervalColumnName(), session.getMaxInactiveInterval() * 1000);
        this.config.getDbPro().update(this.config.getTableName(), this.config.getSessionIdColumnName(), sessionRecord);
    }

    @Override
    public Hashtable<String, BTGSession> getSessions() {
        initDb();

        Hashtable<String, BTGSession> sessions = new Hashtable<>();
        List<Record> records = this.config.getDbPro().find("select * from " + this.config.getTableName());
        for (Record record : records) {
            byte[] sessionBytes = record.getBytes(this.config.getSessionObjColumnName());
            BTGSession session = SerializableUtil.bytes2obj(sessionBytes);
            sessions.put(session.getId(), session);
        }
        return sessions;
    }

    @Override
    public void clearTimeout() {
        initDb();

        //for (BTGSession session : this.getSessions().values()) {
        //    if (session.isInvalidate()) {
        //        this.deleteSession(session.getId());
        //    }
        //}
        this.config.getDbPro().update("delete from " + this.config.getTableName() + " where " + this.config.getSessionLastActiveTimeColumnName() + "  + " + this.config.getSessionMaxInactiveIntervalColumnName() + " < " + System.currentTimeMillis());
    }

    @Override
    public void clear() {
        initDb();

        this.config.getDbPro().update("truncate table " + this.config.getTableName());
    }

    /**
     * 初始化数据库（检查表是否存在，不存在自动创建）
     */
    private void initDb() {
        try {
            if (isInit) {
                return;
            }

            DbPro dbPro = this.config.getDbPro();

            Connection conn = dbPro.getConfig().getConnection();
            DatabaseMetaData meta = conn.getMetaData();
            ResultSet rs = meta.getTables(conn.getCatalog(), conn.getSchema(), this.config.getTableName(), new String[]{"TABLE"});
            boolean tbIsExists = rs.next();

            if (!tbIsExists) {
                String pkgName = BTGSession.class.getPackage().getName();
                String dirName = pkgName.replace(".", "/");

                String dialectStr = "mysql";
                Dialect dialect = dbPro.getConfig().getDialect();
                if (dialect instanceof OracleDialect) {
                    dialectStr = "oracle";
                }

                String sqlFileResource = "/" + dirName + "/BTGDBSessionDao_" + dialectStr + ".sql";
                InputStream is = this.getClass().getResourceAsStream(sqlFileResource);
                if (EmptyUtil.isEmpty(is)) {//非jar包环境
                    is = new FileInputStream(this.getClass().getResource(sqlFileResource).getFile());
                }
                String sql = FileUtil.readToString(is);

                sql = sql.replaceAll("\\s+", " ");

                String[] sqls = sql.split(";");

                for (String s : sqls) {
                    s = s.trim();
                    if (EmptyUtil.isEmpty(s)) {
                        continue;
                    }
                    dbPro.update(s);
                }
            }

            isInit = true;
        } catch (Exception e) {
            throw new BtgRuntimeException("db init error", e);
        }
    }
}