package cn.zcltd.btg.set.util;

import javax.crypto.*;
import javax.crypto.spec.SecretKeySpec;
import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;

/**
 * 3DES加密组件
 */
public class Encryption {

    public static final String ENCODING_DEFAULT = "UTF-8";
    public static final String ENCODING_UTF8 = "UTF-8";
    public static final String ENCODING_GBK = "GBK";

    // //////////////////////////////////////////////////////////
    // 3DES加密解密
    // //////////////////////////////////////////////////////////
    private static final String TDES_KEY_ALGORITHM = "DESede";

    /**
     * 算法名称/加密模式(共有以下四种工作模式)/填充方式(共有3中填充模式)
     * <pre>
     * ECB：电子密码本模式
     * CBC：加密分组链接模式
     * CFB：加密反馈模式
     * OFB：输出反馈模式
     * </pre>
     * <pre>
     * NoPadding
     * PKCS5Padding
     * ISO10126Padding
     * </pre>
     */
    // public static final String TDES_CIPHER_ALGORITHM = "DESede/CBC/NoPadding";
    public static final String TDES_CIPHER_ALGORITHM = "DESede";

    /**
     * 生成TDES秘钥
     *
     * @return TDES秘钥
     */
    public static byte[] generatorTDESKey() {
        KeyGenerator kg = null;
        try {
            kg = KeyGenerator.getInstance(TDES_KEY_ALGORITHM);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        SecureRandom secureRandom = new SecureRandom();
        kg.init(secureRandom);
        SecretKey secretKey = kg.generateKey();
        return secretKey.getEncoded();
    }

    /**
     * 生成TDES秘钥
     *
     * @param key 秘钥
     * @return TDES秘钥
     */
    public static byte[] generatorTDESKey(byte[] key) {
        KeyGenerator kg = null;
        try {
            kg = KeyGenerator.getInstance(TDES_KEY_ALGORITHM);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        SecureRandom secureRandom = new SecureRandom(key);
        kg.init(secureRandom);
        SecretKey secretKey = kg.generateKey();
        return secretKey.getEncoded();
    }

    /**
     * TDES加密
     *
     * @param data 待加密数据
     * @param key  密钥
     * @return 加密后的数据
     */
    public static byte[] encryptTDES(byte[] data, byte[] key) {
        byte[] result = null;
        SecretKey deskey = new SecretKeySpec(key, TDES_KEY_ALGORITHM);
        Cipher cipher = null;
        try {
            cipher = Cipher.getInstance(TDES_CIPHER_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, deskey);
            return cipher.doFinal(data);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (NoSuchPaddingException e) {
            e.printStackTrace();
        } catch (IllegalBlockSizeException e) {
            e.printStackTrace();
        } catch (BadPaddingException e) {
            e.printStackTrace();
        } catch (InvalidKeyException e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * TDES加密
     *
     * @param data     需要加密的字符串
     * @param key      秘钥
     * @param encoding 需要加密的数据和秘钥的解析编码
     * @return 加密后的十六进制字符串
     */
    public static String encryptTDESToString(String data, String key, String encoding) {
        String result = null;
        try {
            return bytesToHexString(encryptTDES(data.getBytes(encoding), hexStringToBytes(key)));
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * TDES加密
     *
     * @param data 需要加密的字符串
     * @param key  秘钥
     * @return 加密后的十六进制字符串
     */
    public static String encryptTDESToString(String data, String key) {
        return encryptTDESToString(data, key, ENCODING_DEFAULT);
    }

    /**
     * TDES解密
     *
     * @param data 待解密数据
     * @param key  密钥
     * @return 解密后的数据
     */
    public static byte[] decryptTDES(byte[] data, byte[] key) {
        byte[] result = null;
        Key deskey = new SecretKeySpec(key, TDES_KEY_ALGORITHM);
        Cipher cipher = null;
        try {
            cipher = Cipher.getInstance(TDES_CIPHER_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, deskey);
            result = cipher.doFinal(data);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (NoSuchPaddingException e) {
            e.printStackTrace();
        } catch (IllegalBlockSizeException e) {
            e.printStackTrace();
        } catch (BadPaddingException e) {
            e.printStackTrace();
        } catch (InvalidKeyException e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * TDES解密
     *
     * @param data     需要解密的十六进制字符串
     * @param key      秘钥
     * @param encoding 需要加密的数据和秘钥的解析编码
     * @return 解密后的原始字符串
     */
    public static String decryptTDESToString(String data, String key, String encoding) {
        String result = null;
        try {
            result = new String(decryptTDES(hexStringToBytes(data), hexStringToBytes(key)), encoding);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * TDES解密
     *
     * @param data 需要解密的十六进制字符串
     * @param key  秘钥
     * @return 解密后的原始字符串
     */
    public static String decryptTDESToString(String data, String key) {
        return decryptTDESToString(data, key, ENCODING_DEFAULT);
    }

    // //////////////////////////////////////////////////////////
    // 十六进制转换
    // //////////////////////////////////////////////////////////
    private final static byte[] HEX_BYTES = "0123456789ABCDEF".getBytes();

    /**
     * 将byte[]转换为十六进制进制字符串
     *
     * @param bytes byte数组
     * @return 十六进制进制字符串
     */
    public static String bytesToHexString(byte[] bytes) {
        int l = bytes.length;
        byte[] buff = new byte[2 * l];
        for (int i = 0; i < l; i++) {
            buff[2 * i] = HEX_BYTES[(bytes[i] >> 4) & 0x0f];
            buff[2 * i + 1] = HEX_BYTES[bytes[i] & 0x0f];
        }
        return new String(buff);
    }

    /**
     * 将十六进制进制字符串转换为byte[]
     *
     * @param hexstr byte数组
     * @return 十六进制进制字符串
     */
    public static byte[] hexStringToBytes(String hexstr) {
        byte[] b = new byte[hexstr.length() / 2];
        int k = 0;
        for (int i = 0, l = b.length; i < l; i++) {
            char c0 = hexstr.charAt(k++);
            char c1 = hexstr.charAt(k++);
            b[i] = (byte) ((parse(c0) << 4) | parse(c1));
        }
        return b;
    }

    /**
     * 将字符转换为整数
     *
     * @param c 需要计算的字符
     * @return 计算后的整数
     */
    private static int parse(char c) {
        if (c >= 'a') return (c - 'a' + 10) & 0x0f;
        if (c >= 'A') return (c - 'A' + 10) & 0x0f;
        return (c - '0') & 0x0f;
    }

    // //////////////////////////////////////////////////////////
    // main测试
    // //////////////////////////////////////////////////////////

    /**
     * 测试主方法
     *
     * @param args args
     * @throws Exception 异常
     */
    public static void main(String[] args) throws Exception {
        String sourceStr = "20170901160633:.*:.*";
        String mykey = "admin";
        byte[] encodeBytes = null;
        String encodeStr = null;
        byte[] decodeBytes = null;
        String decodeStr = null;
        String key = null;

        // 原始数据
        System.out.println("原始数据：\t" + sourceStr);
        System.out.println("----------------------------------------");

        /*
         * 3DES带秘钥加密解密
         */
        key = bytesToHexString(generatorTDESKey(hexStringToBytes(mykey)));
        System.out.println("3DES秘钥：\t" + key);

        encodeBytes = encryptTDES(sourceStr.getBytes(ENCODING_UTF8), hexStringToBytes(key));
        encodeStr = bytesToHexString(encodeBytes);
        System.out.println("3DES加密后1：\t" + encodeStr);
        encodeStr = encryptTDESToString(sourceStr, key);
        System.out.println("3DES加密后2：\t" + encodeStr);
        encodeStr = encryptTDESToString(sourceStr, key, ENCODING_UTF8);
        System.out.println("3DES加密后3：\t" + encodeStr);

        decodeBytes = decryptTDES(hexStringToBytes(encodeStr), hexStringToBytes(key));
        decodeStr = new String(decodeBytes, ENCODING_UTF8);
        System.out.println("3DES解密后1：\t" + decodeStr);
        decodeStr = decryptTDESToString(encodeStr, key);
        System.out.println("3DES解密后2：\t" + decodeStr);
        decodeStr = decryptTDESToString(encodeStr, key, ENCODING_UTF8);
        System.out.println("3DES解密后3：\t" + decodeStr);
        System.out.println("----------------------------------------");
    }
}