package cn.zcltd.btg.set.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * util:File
 */
public class FileUtil {
    private static final Logger log = LoggerFactory.getLogger(FileUtil.class);

    public static final String ENCODING_UTF_8 = "UTF-8";
    public static final String ENCODING_GBK = "GBK";

    /**
     * 从指定文件中读取内容
     *
     * @param file 文件
     * @return 内容
     */
    public static byte[] read(File file) {
        if (!file.exists()) {
            throw new RuntimeException("file not exists：" + file.getAbsolutePath());
        }
        if (file.isDirectory()) {
            throw new RuntimeException("file is a folder：" + file.getAbsolutePath());
        }

        byte[] buffer;
        try {
            FileInputStream in = new FileInputStream(file);
            int size = in.available();
            buffer = new byte[size];
            int a = in.read(buffer);
            if (a == -1) {
                throw new RuntimeException("file is empty：" + file.getAbsolutePath());
            }
            in.close();
        } catch (Exception e) {
            throw new RuntimeException("read file error：" + file.getAbsolutePath(), e);
        }

        return buffer;
    }

    /**
     * 从指定文件中读取内容
     *
     * @param filePath 文件路径
     * @return 内容
     */
    public static byte[] read(String filePath) {
        File file = new File(filePath);
        return read(file);
    }

    /**
     * 从指定文件中读取内容为字符串
     *
     * @param file 文件
     * @return 内容
     */
    public static String readToString(File file) {
        return new String(read(file));
    }

    /**
     * 从指定文件中读取内容为字符串
     *
     * @param file     文件
     * @param encoding 字符编码
     * @return 内容
     */
    public static String readToString(File file, String encoding) {
        byte[] bytes = read(file);
        try {
            return new String(bytes, encoding);
        } catch (Exception e) {
            throw new RuntimeException("read file error：" + file.getAbsolutePath(), e);
        }
    }

    /**
     * 从指定文件中读取内容为字符串
     *
     * @param filePath 文件路径
     * @return 内容
     */
    public static String readToString(String filePath) {
        return new String(read(filePath));
    }

    /**
     * 从指定文件中读取内容为字符串
     *
     * @param filePath 文件路径
     * @param encoding 字符编码
     * @return 内容
     */
    public static String readToString(String filePath, String encoding) {
        byte[] bytes = read(filePath);
        try {
            return new String(bytes, encoding);
        } catch (Exception e) {
            throw new RuntimeException("read file error：" + filePath, e);
        }
    }

    /**
     * 将内容写入到指定文件
     *
     * @param file    文件
     * @param content 内容
     */
    public static void write(File file, byte[] content) {
        if (file.isDirectory()) {
            throw new RuntimeException("file is a folder：" + file.getAbsolutePath());
        }
        try {
            if (!file.exists()) {
                boolean isCreate = file.createNewFile();
                if (!isCreate) {
                    throw new RuntimeException("create file error：" + file.getAbsolutePath());
                }
            }
            FileOutputStream fos = new FileOutputStream(file);
            fos.write(content);
            fos.close();
        } catch (Exception e) {
            throw new RuntimeException("file write error：" + file.getAbsolutePath(), e);
        }
    }

    /**
     * 将内容写入到指定文件
     *
     * @param file    文件
     * @param content 内容
     */
    public static void write(File file, String content) {
        write(file, content.getBytes());
    }

    /**
     * 将内容写入到指定文件
     *
     * @param file     文件
     * @param content  内容
     * @param encoding 编码
     */
    public static void write(File file, String content, String encoding) {
        byte[] bytes;
        try {
            bytes = content.getBytes(encoding);
        } catch (Exception e) {
            throw new RuntimeException("write file error：" + file.getAbsolutePath(), e);
        }
        write(file, bytes);
    }

    /**
     * 将内容写入到指定文件
     *
     * @param filePath 文件路径
     * @param content  内容
     */
    public static void write(String filePath, byte[] content) {
        File file = new File(filePath);
        write(file, content);
    }

    /**
     * 将内容写入到指定文件
     *
     * @param filePath 文件路径
     * @param content  内容
     */
    public static void write(String filePath, String content) {
        File file = new File(filePath);
        write(file, content.getBytes());
    }

    /**
     * 将内容写入到指定文件
     *
     * @param filePath 文件路径
     * @param content  内容
     * @param encoding 编码
     */
    public static void write(String filePath, String content, String encoding) {
        File file = new File(filePath);
        byte[] bytes;
        try {
            bytes = content.getBytes(encoding);
        } catch (Exception e) {
            throw new RuntimeException("write file error：" + filePath, e);
        }
        write(file, bytes);
    }

    /**
     * (vip)查找指定baseDirPath路径下与matchesStr匹配的文件
     *
     * @param baseDirPath  起始文件夹路径
     * @param matchesStr   文件名称(不包含路径)匹配规则(正则表达式)
     * @param scanChildDir 是否扫描子文件夹
     * @return List
     */
    public static List<File> findFiles(String baseDirPath, String matchesStr, boolean scanChildDir) {
        try {
            List<File> files = new ArrayList<File>();

            File fileBaseDir = new File(baseDirPath);
            if (!fileBaseDir.exists()) {
                log.debug("directory not exists:" + baseDirPath);
                return null;
            }
            if (!fileBaseDir.isDirectory()) {
                log.debug("not a directory:" + baseDirPath);
                return null;
            }

            File[] filelist = fileBaseDir.listFiles();
            if (filelist == null) {
                return null;
            }

            for (File file : filelist) {
                if (file.isDirectory()) {
                    if (scanChildDir) {
                        files.addAll(findFiles(file.getPath(), matchesStr, scanChildDir));
                    }
                    continue;
                }

                if (file.getName().matches(matchesStr)) {
                    files.add(file.getAbsoluteFile());
                }
            }

            return files;
        } catch (Exception e) {
            log.debug(e.getMessage(), e);
        }

        return null;
    }

    /**
     * 查找指定baseDirPath路径下(包含子文件夹)与matchesStr匹配的文件
     *
     * @param baseDirPath 起始文件夹路径
     * @param matchesStr  文件名称(不包含路径)匹配规则(正则表达式)
     * @return List
     */
    public static List<File> findFiles(String baseDirPath, String matchesStr) {
        return findFiles(baseDirPath, matchesStr, true);
    }

    /**
     * 查找指定路径下(包含子文件夹)与matchesStr匹配的文件
     *
     * @param baseDirPath 起始文件夹路径
     * @return List
     */
    public static List<File> findFiles(String baseDirPath) {
        return findFiles(baseDirPath, ".*", true);
    }
}