package cn.zcltd.btg.set.core;

import cn.zcltd.btg.set.ParamType;
import cn.zcltd.btg.set.exception.BtgSetRuntimerException;
import cn.zcltd.btg.sutil.EmptyUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Param参数类型转换
 */
public class ParamConvert {
    private static final Logger log = LoggerFactory.getLogger(ParamConvert.class);

    public static String getString(Param param) {
        return getString(param, null);
    }

    public static String getString(Param param, String defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            return value;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }

    public static Boolean getBoolean(Param param) {
        return getBoolean(param, null);
    }

    public static Boolean getBoolean(Param param, Boolean defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            return Boolean.valueOf(value);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }

    public static Number getNumber(Param param) {
        return getNumber(param, null);
    }

    public static Number getNumber(Param param, Number defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            if (param.getType().equals(ParamType.Boolean)) {
                throw new BtgSetRuntimerException("Boolean not cast to Number：" + param.getPath());
            }
            if (param.getType().equals(ParamType.String)) {
                throw new BtgSetRuntimerException("String not cast to Number：" + param.getPath());
            }
            if (param.getType().equals(ParamType.Integer)) {
                return getInteger(param);
            }
            if (param.getType().equals(ParamType.Long)) {
                return getLong(param);
            }
            if (param.getType().equals(ParamType.Float)) {
                return getFloat(param);
            }
            if (param.getType().equals(ParamType.Double)) {
                return getDouble(param);
            }
            if (param.getType().equals(ParamType.BigDecimal)) {
                return getBigDecimal(param);
            }
            if (param.getType().equals(ParamType.Date)) {
                throw new BtgSetRuntimerException("Date not cast to Number：" + param.getPath());
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }

    public static Integer getInteger(Param param) {
        return getInteger(param, null);
    }

    public static Integer getInteger(Param param, Integer defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            if (param.getType().equals(ParamType.Integer)) {
                return Integer.valueOf(param.getValue());
            }

            throw new BtgSetRuntimerException("dataType is not Number：" + param.getPath() + "[" + param.getType().getName() + "]");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }

    public static Long getLong(Param param) {
        return getLong(param, null);
    }

    public static Long getLong(Param param, Long defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            if (param.getType().equals(ParamType.Long)) {
                return Long.valueOf(param.getValue());
            }

            throw new BtgSetRuntimerException("dataType is not Number：" + param.getPath() + "[" + param.getType().getName() + "]");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }

    public static Float getFloat(Param param) {
        return getFloat(param, null);
    }

    public static Float getFloat(Param param, Float defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            if (param.getType().equals(ParamType.Float)) {
                return Float.valueOf(param.getValue());
            }

            throw new BtgSetRuntimerException("dataType is not Number：" + param.getPath() + "[" + param.getType().getName() + "]");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }

    public static Double getDouble(Param param) {
        return getDouble(param, null);
    }

    public static Double getDouble(Param param, Double defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            if (param.getType().equals(ParamType.Double)) {
                return Double.valueOf(param.getValue());
            }

            throw new BtgSetRuntimerException("dataType is not Number：" + param.getPath() + "[" + param.getType().getName() + "]");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }

    public static BigDecimal getBigDecimal(Param param) {
        return getBigDecimal(param, null);
    }

    public static BigDecimal getBigDecimal(Param param, BigDecimal defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            if (param.getType().equals(ParamType.BigDecimal)) {
                return new BigDecimal(param.getValue());
            }

            throw new BtgSetRuntimerException("dataType is not Number：" + param.getPath() + "[" + param.getType().getName() + "]");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }

    public static Date getDate(Param param) {
        return getDate(param, "");
    }

    public static Date getDate(Param param, Date defaultValue) {
        return getDate(param, "", defaultValue);
    }

    public static Date getDate(Param param, String datePattern) {
        return getDate(param, datePattern, null);
    }

    public static Date getDate(Param param, String datePattern, Date defaultValue) {
        try {
            if (null == param) return defaultValue;
            String value = param.getValue();
            if (EmptyUtil.isEmpty(value)) return defaultValue;

            if (param.getType().equals(ParamType.Date)) {
                if (EmptyUtil.isEmpty(datePattern) && EmptyUtil.isNotEmpty(param.getDatePattern())) {
                    datePattern = param.getDatePattern();
                }
                if (EmptyUtil.isEmpty(datePattern)) {
                    int length = param.getValue().length();
                    if (length == 10) {
                        datePattern = "yyyy-MM-dd";
                    } else if (length == 8) {
                        datePattern = "HH:mm:ss";
                    } else if (length == 19) {
                        datePattern = "yyyy-MM-dd HH:mm:ss";
                    } else if (length == 17) {
                        datePattern = "yyyyMMddHHmmssSSS";
                    }
                }
                if (EmptyUtil.isEmpty(datePattern)) {
                    throw new BtgSetRuntimerException("datePattern error：" + param.getPath());
                }
                try {
                    return new SimpleDateFormat(datePattern).parse(param.getValue());
                } catch (ParseException e) {
                    throw new BtgSetRuntimerException(e.getMessage(), e);
                }
            }

            throw new BtgSetRuntimerException("dataType is not Number：" + param.getPath() + "[" + param.getType().getName() + "]");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return defaultValue;
    }
}