package cn.zcltd.btg.set.sourcemanager;

import cn.zcltd.btg.set.core.Config;
import cn.zcltd.btg.set.exception.BtgSetRuntimerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 配置加载器接口
 * 实现此接口以实现从不同渠道获取配置信息
 */
public abstract class SourceManager {
    protected final Logger log = LoggerFactory.getLogger(this.getClass());

    private Map<String, Config> cache = new HashMap<>();
    protected String name;//名称
    private boolean devMode = false;//开发模式：当为开发模式时，不适用缓存，直接获取最新数据

    public SourceManager(String name) {
        this.name = name;
    }

    public Map<String, Config> getCache() {
        return cache;
    }

    public void setCache(Map<String, Config> cache) {
        this.cache = cache;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public boolean isDevMode() {
        return devMode;
    }

    public void setDevMode(boolean devMode) {
        this.devMode = devMode;
    }

    /**
     * 初始化SourceManager
     *
     * @param devModel devModel
     */
    public abstract void init(boolean devModel);

    /**
     * 获取所有配置（获取cache）
     *
     * @return 配置集合
     */
    public List<Config> useAll() {
        return new ArrayList<>(this.cache.values());
    }

    /**
     * 获取指定id配置（优先获取cache）
     *
     * @param configId configId
     * @return 配置
     */
    public Config use(String configId) {
        Config config;

        if (!isDevMode()) {
            config = this.cache.get(configId);
            if (null != config) {
                return config;
            }
        }

        config = this.load(configId);

        if (null == config) {
            throw new BtgSetRuntimerException(configId + " not exists");
        }

        this.cache.put(config.getId(), config);

        return config;
    }

    /**
     * 获取所有配置
     *
     * @return 配置集合
     */
    public abstract List<Config> loadAll();

    /**
     * 获取指定id配置
     *
     * @param configId configId
     * @return 配置
     */
    public abstract Config load(String configId);

    /**
     * 保存配置
     *
     * @param config 配置
     */
    public abstract void save(Config config);
}