package cn.zcltd.btg.validq;

import cn.zcltd.btg.validq.exception.ValidQException;

/**
 * 验证处理器
 * 该类提供了默认的处理器实现，自定义Validator可以继承此类而不需要实现接口重写所有实现
 * 改实现提供了一些常用字段供使用，并且这些字段的set方法是链式的
 */
public abstract class ValidatorHandler<T> implements Validator<T> {
    private String field;//字段（若有多级时如“filed_root.field_parent[0].field_son”）
    private String shortField;//短字段（多级时没有上级，如“field_son”）
    private String fieldName;//字段名称（若有多级时如“跟节点.父列表[第1条].子节点”）
    private String shortFieldName;//短字段名称（多级时没有上级，如“子节点”）
    private String errorCode;//错误代码（来源于Validator）
    private String errorMsg;//错误信息

    public ValidatorHandler() {
    }

    public ValidatorHandler(String field, String fieldName) {
        this.field = field;
        this.shortField = field;
        this.fieldName = fieldName;
        this.shortFieldName = fieldName;
    }

    public ValidatorHandler(String field, String fieldName, String errorMsg) {
        this.field = field;
        this.shortField = field;
        this.fieldName = fieldName;
        this.shortFieldName = fieldName;
        this.errorMsg = errorMsg;
    }

    @Override
    public int index() {
        return 0;
    }

    @Override
    public boolean accept(ValidatorContext context, T target) {
        return true;
    }

    @Override
    public boolean validate(ValidatorContext context, T target) {
        return true;
    }

    @Override
    public void onException(Exception e, ValidatorContext context, T target) {
        throw new ValidQException(e);
    }

    /**
     * 获取默认提示信息（第一个占位符为字段名称）
     *
     * @param errorMsg 包含占位符的错误信息
     * @return 预处理错误信息
     */
    protected String getDefaultMsg(String errorMsg) {
        String formatErrorMsg = this.getErrorMsg();
        if (null == formatErrorMsg || formatErrorMsg.trim().length() == 0) {
            formatErrorMsg = errorMsg;
        }
        if (null != getFieldName() && getFieldName().trim().length() > 0) {
            formatErrorMsg = String.format(formatErrorMsg, getFieldName());
        } else if (null != getShortFieldName() && getShortFieldName().trim().length() > 0) {
            formatErrorMsg = String.format(formatErrorMsg, getShortFieldName());
        } else if (null != getField() && getField().trim().length() > 0) {
            formatErrorMsg = String.format(formatErrorMsg, getField());
        } else if (null != getShortField() && getShortField().trim().length() > 0) {
            formatErrorMsg = String.format(formatErrorMsg, getShortField());
        } else {
            throw new ValidQException("fieldName,shortFieldName,field,shortField,errorMsg has at least one");
        }
        if (null != getErrorCode() && getErrorCode().trim().length() > 0) {
            formatErrorMsg = String.format("%s:" + formatErrorMsg, getErrorCode());
        }
        return formatErrorMsg;
    }

    public String getField() {
        return field;
    }

    public ValidatorHandler setField(String field) {
        this.field = field;
        return this;
    }

    public String getShortField() {
        return shortField;
    }

    public ValidatorHandler setShortField(String shortField) {
        this.shortField = shortField;
        return this;
    }

    public String getFieldName() {
        return fieldName;
    }

    public ValidatorHandler setFieldName(String fieldName) {
        this.fieldName = fieldName;
        return this;
    }

    public String getShortFieldName() {
        return shortFieldName;
    }

    public ValidatorHandler setShortFieldName(String shortFieldName) {
        this.shortFieldName = shortFieldName;
        return this;
    }

    public String getErrorCode() {
        return errorCode;
    }

    public ValidatorHandler setErrorCode(String errorCode) {
        this.errorCode = errorCode;
        return this;
    }

    public String getErrorMsg() {
        return errorMsg;
    }

    public ValidatorHandler setErrorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
        return this;
    }
}