package cn.zhxu.bp.auth;

import cn.zhxu.bp.enums.SysConst;
import cn.zhxu.data.Array;
import cn.zhxu.data.Mapper;
import cn.zhxu.bp.RespException;
import cn.zhxu.bp.enums.UserType;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.ToString;

import java.util.*;

/**
 * @author Troy.Zhou @ 2022/8/2 23:09
 */
@Getter
@AllArgsConstructor
@ToString
public class Principal {

    /**
     * 当前登录的平台 ID
     */
    private final int saasId;

    /**
     * 当前登录的用户
     */
    private final User user;

    /**
     * @return 是否为上帝账号来访
     */
    public boolean isSysGodVisit() {
        return user != null && user.isSysGodVisit(saasId);
    }

    /**
     * @return 是否为平台顶级账号或子帝账号来访
     */
    public boolean isTopOrSubGodVisit() {
        return user != null && user.isTopOrSubGodVisit(saasId);
    }

    /**
     * @return 当前访问的用户（非空）
     */
    public User requireUser() {
        if (user == null) {
            throw RespException.forbidden("Current User is Null");
        }
        return user;
    }

    /**
     * @return 当前访问的用户
     */
    public Optional<User> optionalUser() {
        return Optional.ofNullable(user);
    }

    public Principal with(Scope scope) {
        if (user == null) {
            return this;
        }
        return new Principal(saasId, user.with(scope));
    }

    public Principal with(int saasId) {
        return new Principal(saasId, user);
    }

    @Getter
    public static class User {

        /**
         * 用户 ID
         */
        private final long id;

        /**
         * 用户编码（全局唯一）
         */
        private final String no;

        /**
         * 所属组织 ID, 0 表示总公司
         */
        private final int groupId;

        /**
         * 数据权限
         */
        private final Scope scope;

        /**
         * 用户名
         */
        private final String name;

        /**
         * 用户类型数值
         */
        private final UserType type;

        /**
         * 用户的系统角色键值
         */
        private final List<String> roleKeys;

        /**
         * 其它属性
         */
        private final Map<String, Object> attrs;

        public User(long id, String no, int groupId, String name, UserType type, List<String> roleKeys, Map<String, Object> attrs) {
            this(id, no, groupId, null, name, type, roleKeys, attrs);
        }

        public User(long id, String no, int groupId, Scope scope, String name, UserType type, List<String> roleKeys, Map<String, Object> attrs) {
            this.id = id;
            this.no = no;
            this.groupId = groupId;
            this.scope = scope;
            this.name = name;
            this.type = type;
            this.roleKeys = roleKeys;
            this.attrs = attrs;
        }

        public User with(Scope scope) {
            return new User(id, no, groupId, scope, name, type, roleKeys, attrs);
        }

        public interface AttrKeys {
            String SYS_GOD = "s";
            String SUB_GOD = "b";
        }

        /**
         * 子帝账号，可跨平台登录
         * @return 是否为子帝账号
         */
        public boolean isSubGodUser() {
            if (type == UserType.SYS) {
                Object value = attrs.get(AttrKeys.SUB_GOD);
                if (value != null) {
                    String s = value.toString().toLowerCase();
                    return Boolean.parseBoolean(s);
                }
            }
            return false;
        }

        /**
         * 上帝账号，用有超管后台的所有权限
         * @return 是否为上帝账号
         */
        public boolean isSysGodUser() {
            if (type == UserType.SYS) {
                Object value = attrs.get(AttrKeys.SYS_GOD);
                if (value != null) {
                    String s = value.toString().toLowerCase();
                    return Boolean.parseBoolean(s);
                }
            }
            return false;
        }

        /**
         * @return 是否为平台顶级账号
         */
        public boolean isTopUser() {
            return type == UserType.TOP;
        }

        /**
         * @return 是否为平台顶级账号或上帝账号来访
         */
        public boolean isSysGodVisit(int saasId) {
            return isSysGodUser() && saasId == SysConst.SYS_SAAS_ID;
        }

        /**
         * @return 是否为平台顶级账号或上帝账号来访
         */
        public boolean isTopOrSubGodVisit(int saasId) {
            return isTopUser() || isSubGodUser() && saasId > SysConst.SYS_SAAS_ID;
        }

        public Scope requireScope() {
            if (scope == null) {
                throw RespException.forbidden("Scope is NULL");
            }
            return scope;
        }

        public static final String ID = "i";
        public static final String NO = "n";
        public static final String GROUP_ID = "g";
        public static final String SCOPE = "s";
        public static final String NAME = "m";
        public static final String TYPE = "t";
        public static final String ROLE_KEYS = "r";
        public static final String ATTRS = "a";

        public static User parse(Mapper mapper) {
            if (mapper == null) {
                return null;
            }
            long id = mapper.getLong(ID);
            String no = mapper.getString(NO);
            int groupId = mapper.getInt(GROUP_ID);
            Mapper scope = mapper.getMapper(SCOPE);
            String name = mapper.getString(NAME);
            int type = mapper.getInt(TYPE);
            Array roleKeys = mapper.getArray(ROLE_KEYS);
            Mapper attr = mapper.getMapper(ATTRS);
            return new User(
                    id,
                    no,
                    groupId,
                    Scope.parse(scope),
                    name,
                    UserType.of(type),
                    roleKeys.toList(String.class),
                    attr.toMap()
            );
        }

        public Map<String, Object> toMap() {
            Map<String, Object> map = new HashMap<>();
            map.put(ID, id);
            map.put(NO, no);
            map.put(NAME, name);
            map.put(GROUP_ID, groupId);
            if (scope != null) {
                map.put(SCOPE, scope.toMap());
            }
            map.put(TYPE, type.getValue());
            map.put(ROLE_KEYS, roleKeys);
            map.put(ATTRS, attrs);
            return map;
        }

        public String idName() {
            return "[" + id + "]" + name;
        }

    }

    public static final String SAAS_ID = "s";
    public static final String USER = "u";

    public static Principal parse(Mapper mapper) {
        if (mapper.has(SAAS_ID)) {
            int saasId = mapper.getInt(SAAS_ID);
            Mapper user = mapper.getMapper(USER);
            return new Principal(saasId, User.parse(user));
        }
        throw new IllegalArgumentException(SAAS_ID);
    }

    public Map<String, Object> toMap() {
        Map<String, Object> map = new HashMap<>();
        map.put(SAAS_ID, saasId);
        if (user != null) {
            map.put(USER, user.toMap());
        }
        return map;
    }

}
