package cn.zhxu.toys.captcha.impl;

import cn.zhxu.toys.captcha.*;
import org.springframework.beans.factory.InitializingBean;

/**
 * @author Troy.Zhou @ 2022/8/16 11:55
 */
public class BaseCaptchaManager implements CaptchaManager, InitializingBean {

    public static final String ATTR_EXPIRE_SECONDS = "expireSeconds";

    /**
     * 验证码生成器
     */
    private CodeGenerator codeGenerator;

    /**
     * 验证码渲染器
     */
    private CodeRenderer codeRenderer;

    /**
     * 验证码持久化服务
     */
    private CaptchaService captchaService;

    /**
     * 验证码有效时间（默认：5分钟）
     */
    private int expireSeconds = 5 * 60;


    @Override
    public void genAndRender(String captchaId, CaptchaAttrs attrs) throws CaptchaException {
        CodeGenerator.CodeResult result = codeGenerator.generate();
        codeRenderer.render(result.getCode(), attrs);
        Integer expireSeconds = attrs.get(ATTR_EXPIRE_SECONDS, Integer.class, this.expireSeconds);
        captchaService.save(captchaId, result.getCheck(), expireSeconds);
    }

    @Override
    public VerifyResult verify(String captchaId, String code) {
        if (code == null) {
            return VerifyResult.INVALID;
        }
        Captcha captcha = captchaService.findCaptcha(captchaId);
        if (captcha == null) {
            return VerifyResult.INVALID;
        }
        if (captcha.isExpired()) {
            return VerifyResult.EXPIRED;
        }
        if (code.equals(captcha.getCode())) {
            captchaService.delete(captchaId);
            return VerifyResult.SUCCESS;
        } else {
            return VerifyResult.INVALID;
        }
    }

    @Override
    public void afterPropertiesSet() {
        if (captchaService == null) {
            throw new IllegalStateException("You must set a captchaService into CaptchaManager");
        }
        if (codeRenderer == null) {
            throw new IllegalStateException("You must set a codeRenderer into CaptchaManager");
        }
        if (codeGenerator == null) {
            codeGenerator = new RandomCharCodeGenerator();
        }
    }

    public CodeGenerator getCodeGenerator() {
        return codeGenerator;
    }

    public void setCodeGenerator(CodeGenerator codeGenerator) {
        this.codeGenerator = codeGenerator;
    }

    public CodeRenderer getCodeRenderer() {
        return codeRenderer;
    }

    public void setCodeRenderer(CodeRenderer codeRenderer) {
        this.codeRenderer = codeRenderer;
    }

    public CaptchaService getCaptchaService() {
        return captchaService;
    }

    public void setCaptchaService(CaptchaService captchaService) {
        this.captchaService = captchaService;
    }

    public int getExpireSeconds() {
        return expireSeconds;
    }

    public void setExpireSeconds(int expireSeconds) {
        this.expireSeconds = expireSeconds;
    }

}
