package cn.zhxu.toys.msg;

import cn.zhxu.toys.cache.CacheService;
import cn.zhxu.toys.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.List;

/**
 * 检验请求IP的短信发送器
 * @since 0.2.9
 * */
public class IPChkMsgSender implements MsgSender {

	private static final Logger log = LoggerFactory.getLogger(IPChkMsgSender.class);

	private MsgSender msgSender;
	
	private ReqIpGetter reqIpGetter;
	
	private CacheService cacheService;
	
	private String cachePrefix = "msgSender";

	private List<String> whiteIpList;

	/**
	 * 每天每个IP每个短信模板最大发送次数
	 */
	private int maxAllowCountPerDay = 50;
	
	/**
	 * 计数周期
	 */
	private long countCycleSeconds = 24 * 60 * 60;
	
	/**
	 * Redis 过期时间：2天
	 */
	private int redisExpireSeconds = 48 * 3600;
	
	/**
	 * 最小发送间隔，默认5秒
	 */
	private int minSendInterval = 5;
	
	
	@Override
	public boolean send(String phone, String tmplName, String... tmplArgs) {
		String ip = reqIpGetter.getIp();
		if (StringUtils.isBlank(ip)) {
			log.warn("拒绝发送 [phone: " + phone + ", tmplName: " + tmplName + ", IP: " + ip +"]");
			return false;
		}
		if (getWhiteIpList().contains(ip)) {
			log.info("发送短信 [phone: " + phone + ", tmplName: " + tmplName + ", IP: " + ip +"]");
			return msgSender.send(phone, tmplName, tmplArgs);
		}
		String cacheKey = cachePrefix + ":" + ip + ":" + tmplName;
		CacheItem item = cacheService.cache(cacheKey, CacheItem.class);
		long now = System.currentTimeMillis() / 1000;
		if (item != null) {
			long timeDiff = now - item.getLastSentTime();
			if (timeDiff < minSendInterval) {
				log.warn("拒绝发送 [phone: " + phone + ", tmplName: " + tmplName + ", IP: " + ip +"]: 请求过于频繁：" + timeDiff);
				return false;
			}
			if (item.getCountTime() < now - countCycleSeconds) {
				item.setCount(0);
				item.setCountTime(now);
			}
			if (item.getCount() >= maxAllowCountPerDay) {
				log.warn("拒绝发送 [phone: " + phone + ", tmplName: " + tmplName + ", IP: " + ip +"]: 最大条数限制：[sent: "
						+ item.getCount() + ", maxAllow: " + maxAllowCountPerDay + "]");
				return false;
			}
		}
		log.info("发送短信 [phone: " + phone + ", tmplName: " + tmplName + ", IP: " + ip +"]");
		boolean success = msgSender.send(phone, tmplName, tmplArgs);
		if (success) {
			if (item == null) {
				item = new CacheItem();
				item.setCountTime(now);
			}
			item.setLastSentTime(now);
			item.setCount(item.getCount() + 1);
			cacheService.cache(cacheKey, redisExpireSeconds, item);
		}
		return success;
	}

	
	public static class CacheItem {
		
		private int count;
		private long lastSentTime;
		private long countTime;
		
		public int getCount() {
			return count;
		}
		
		public void setCount(int count) {
			this.count = count;
		}

		public long getLastSentTime() {
			return lastSentTime;
		}

		public void setLastSentTime(long lastSentTime) {
			this.lastSentTime = lastSentTime;
		}

		public long getCountTime() {
			return countTime;
		}

		public void setCountTime(long countTime) {
			this.countTime = countTime;
		}
		
	}


	public MsgSender getMsgSender() {
		return msgSender;
	}

	public void setMsgSender(MsgSender msgSender) {
		this.msgSender = msgSender;
	}

	public ReqIpGetter getReqIpGetter() {
		return reqIpGetter;
	}

	public void setReqIpGetter(ReqIpGetter reqIpGetter) {
		this.reqIpGetter = reqIpGetter;
	}

	public CacheService getCacheService() {
		return cacheService;
	}

	public void setCacheService(CacheService cacheService) {
		this.cacheService = cacheService;
	}

	public String getCachePrefix() {
		return cachePrefix;
	}

	public void setCachePrefix(String cachePrefix) {
		this.cachePrefix = cachePrefix;
	}

	public int getMaxAllowCountPerDay() {
		return maxAllowCountPerDay;
	}

	public void setMaxAllowCountPerDay(int maxAllowCountPerDay) {
		this.maxAllowCountPerDay = maxAllowCountPerDay;
	}

	public long getCountCycleSeconds() {
		return countCycleSeconds;
	}

	public void setCountCycleSeconds(long countCycleSeconds) {
		this.countCycleSeconds = countCycleSeconds;
	}

	public int getRedisExpireSeconds() {
		return redisExpireSeconds;
	}

	public void setRedisExpireSeconds(int redisExpireSeconds) {
		this.redisExpireSeconds = redisExpireSeconds;
	}

	public int getMinSendInterval() {
		return minSendInterval;
	}

	public void setMinSendInterval(int minSendInterval) {
		this.minSendInterval = minSendInterval;
	}

	public List<String> getWhiteIpList() {
		return whiteIpList != null ? whiteIpList : Collections.emptyList();
	}

	public void setWhiteIpList(List<String> whiteIpList) {
		this.whiteIpList = whiteIpList;
	}

}
